#include "sessionparser.h"

#include "keyring.h"
#include "sessionsigner.h"
#include "sessionutils.h"

#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/generic/strbuf.h>
#include <util/string/cast.h>

#include <unordered_map>

namespace NPassport::NAuth {
    enum ESessFlags {
        sessSafe = 0x01,
        sessSuspicious = 0x02,
        sessStress = 0x100
    };

    // session k-v fields
    static const TString SESS_USER_AGENT = "0";
    static const TString SESS_LOGIN_ID = "1";
    static const TString SESS_ENVIRONMENT = "2";
    static const TString SESS_EXT_USERIP = "100";
    static const TString SESS_EXT_AUTH_TS = "101";
    static const TString SESS_EXT_UPDATE_TS = "102";
    static const TString SESS_EXT_LEGACY_FLAG = "499";

    // ==================== environment mapping ================
    static const TString SESS_ENV_PROD = "1";
    static const TString SESS_ENV_TEST = "2";
    static const TString SESS_ENV_PROD_YATEAM = "3";
    static const TString SESS_ENV_TEST_YATEAM = "4";
    static const TString SESS_ENV_LOAD = "5";

    static std::optional<EEnvironmentType> GetEnvironmentType(const TStringBuf env) {
        if (env == SESS_ENV_PROD) {
            return EEnvironmentType::Production;
        }
        if (env == SESS_ENV_TEST) {
            return EEnvironmentType::Testing;
        }
        if (env == SESS_ENV_PROD_YATEAM) {
            return EEnvironmentType::ProductionYateam;
        }
        if (env == SESS_ENV_TEST_YATEAM) {
            return EEnvironmentType::TestingYateam;
        }
        if (env == SESS_ENV_LOAD) {
            return EEnvironmentType::Load;
        }

        return {};
    }

    static const TString& GetEnvironmentFromType(EEnvironmentType env) {
        switch (env) {
            case EEnvironmentType::Production:
                return SESS_ENV_PROD;
            case EEnvironmentType::Testing:
                return SESS_ENV_TEST;
            case EEnvironmentType::ProductionYateam:
                return SESS_ENV_PROD_YATEAM;
            case EEnvironmentType::TestingYateam:
                return SESS_ENV_TEST_YATEAM;
            case EEnvironmentType::Load:
                return SESS_ENV_LOAD;
        }
    }

    enum EUserFlags {
        userLite = 0x01,
        userHavePwd = 0x02,
        userStaff = 0x100,
        userBetatester = 0x200,
        userGlogouted = 0x400,
        userExtGlogouted = 0x800,
        userInternalAuth = 0x2000,
        userExternalAuth = 0x4000,
        userScholar = 0x8000,
    };

    // user k-v fields
    static const TString USER_LANG = "0";
    static const TString USER_SOCIALID = "1";
    static const TString USER_LOGIN_DELTA = "2";

    static const char DOT('.');
    static const char COLON(':');
    static const char PIPELINE('|');

    static const TString STRDOT(".");
    static const TString STRCOLON(":");
    static const TString STRCAP("^");
    static const TString STRPIPE("|");
    static const TString NOAUTH("noauth");

    // =================== error message dictionary ================
    static const std::unordered_map<NSessionCodes::ESessionError, TString> SESSION_CODES = {
        {NSessionCodes::OK, "no error"},
        {NSessionCodes::BAD_COOKIE_SIZE, "cookie size is out of proper range"},
        {NSessionCodes::BAD_COOKIE_BODY, "cookie body doesn't corespond to proper format"},
        {NSessionCodes::INVALID_SOURCE, "the source for the object to copy is in bad state"},
        {NSessionCodes::BAD_COOKIE_TS, "cookie timestamp field is corrupt"},
        {NSessionCodes::HOST_DONT_MATCH, "hostname specified doesn't belong to cookie domain"},
        {NSessionCodes::NO_DATA_KEYSPACE, "there is no data for given keyspace"},
        {NSessionCodes::KEY_NOT_FOUND, "key with specified id isn't found"},
        {NSessionCodes::KEYSPACE_EMPTY, "no keys are defined in specified keyspace"},
        {NSessionCodes::BAD_SIGN, "signature has bad format or is broken"},
        {NSessionCodes::BAD_DATA_SIZE, "size of input data is either too small or too big"},
        {NSessionCodes::TOO_OLD_COOKIE, "cookie is too old and cannot be checked"},
        {NSessionCodes::WRONG_ENVIRONMENT, "cookie was got in wrong environment "
                                           "(production/intranet_production/testing)"},
    };

    static const TString UNKNOWN_ERROR = "unknown error";
    const TString& TSessionParser::ErrAsString(const NSessionCodes::ESessionError err) {
        auto it = SESSION_CODES.find(err);
        if (it == SESSION_CODES.end()) {
            TLog::Error() << "Absent error message requested: " << (int)err;
            return UNKNOWN_ERROR;
        }

        return it->second;
    }

    TSessionParser::TSessionParser(const TSessionSigner& sessSigner, EEnvironmentType env)
        : SessSigner_(sessSigner)
        , Environment_(env)
    {
    }

    // parse authid field
    time_t TSessionParser::ParseAuthIdTs(TStringBuf authid_time) {
        if (authid_time.size() < 3) {
            return 0;
        }
        time_t t = 0;
        TryIntFromString<10>(authid_time.Chop(3), t);
        return t;
    }

    bool TSessionParser::ParseAuthId(TSession::TAuthid& authid) {
        TStringBuf buf(authid.Str);
        TStringBuf time = buf.NextTok(':');
        TStringBuf ip = buf.NextTok(':');
        TStringBuf host = buf.NextTok(':');

        if (time.empty() ||
            ip.empty() ||
            host.empty() ||
            !NUtils::DigitsOnly(time) ||
            !NUtils::IsBase64url(ip) ||
            !NUtils::HexDigitsOnly(host))
        {
            return false;
        }

        authid.Time = time;
        authid.Ip = ip;
        authid.Host = host;
        authid.Ts = ParseAuthIdTs(authid.Time);

        return true;
    }

    TString TSessionParser::MakeFlags(const TSession::TSessionData& data) {
        unsigned flags = sessSafe;

        if (data.Suspicious) {
            flags |= sessSuspicious;
        }
        if (data.Stress) {
            flags |= sessStress;
        }

        char buf[20];
        snprintf(buf, 20, "%x", flags);

        return TString(buf);
    }

    TString TSessionParser::MakeFlags(const TSession::TUserData& data) {
        unsigned flags = 0;

        if (data.Lite) {
            flags |= userLite;
        }
        if (data.HavePassword) {
            flags |= userHavePwd;
        }
        if (data.Staff) {
            flags |= userStaff;
        }
        if (data.Betatester) {
            flags |= userBetatester;
        }
        if (data.Glogouted) {
            flags |= userGlogouted;
        }
        if (data.InternalAuth) {
            flags |= userInternalAuth;
        }
        if (data.ExternalAuth) {
            flags |= userExternalAuth;
        }
        if (data.ExtGlogouted) {
            flags |= userExtGlogouted;
        }
        if (data.Scholar) {
            flags |= userScholar;
        }

        char buf[20];
        snprintf(buf, 20, "%x", flags);

        return TString(buf);
    }

    TString TSessionParser::MakeExt(const TSession::TSessionData& data, TSession::ECategory category) {
        TString ext;

        if (!data.UserAgent.empty()) {
            NUtils::Append(ext, STRDOT, SESS_USER_AGENT, STRCOLON, data.UserAgent);
        }

        if (!data.LoginId.empty()) {
            NUtils::Append(ext, STRDOT, SESS_LOGIN_ID, STRCOLON, NUtils::Bin2base64url(data.LoginId));
        }

        if (data.Environment) {
            NUtils::Append(ext, STRDOT, SESS_ENVIRONMENT, STRCOLON, GetEnvironmentFromType(*data.Environment));
        }

        if (!data.ExtUserip.empty()) {
            NUtils::Append(ext, STRDOT, SESS_EXT_USERIP, STRCOLON, data.ExtUserip);
        }

        if (!data.ExtAuthTs.empty()) {
            NUtils::Append(ext, STRDOT, SESS_EXT_AUTH_TS, STRCOLON, data.ExtAuthTs);
        }

        if (!data.ExtUpdateTs.empty()) {
            NUtils::Append(ext, STRDOT, SESS_EXT_UPDATE_TS, STRCOLON, data.ExtUpdateTs);
        }

        if (category == TSession::ECategory::Legacy) {
            NUtils::Append(ext, STRDOT, SESS_EXT_LEGACY_FLAG, STRCOLON, "1");
        }

        return ext;
    }

    TString TSessionParser::MakeExt(const TSession::TUserData& data) {
        TString ext;

        if (!data.Lang.empty() && data.Lang != "1") {
            ext.append(".0:").append(data.Lang);
        }

        if (!data.SocialId.empty()) {
            ext.append(".1:").append(data.SocialId);
        }

        if (data.LoginDelta != 0) {
            ext.append(".2:").append(IntToString<10>(data.LoginDelta));
        }

        return ext;
    }

    TString TSessionParser::MakeBodyV3(const TSession& src, const TString& strtime, TSession::ECategory category) {
        const TString def_idx = IntToString<16>(src.DefaultIdx());
        TString body;

        size_t reserve = 32 * src.Users_.Count();
        NUtils::AppendExt(body, reserve, "3:", strtime, STRDOT, src.Ttl_, STRDOT, def_idx, STRDOT, src.AuthId(), STRDOT, MakeFlags(src.Data_), MakeExt(src.Data_, category));

        for (unsigned id = 0; id < src.Users_.Count(); ++id) {
            const TSession::TUserData& user = src.Users_.Get(id);
            NUtils::Append(body, STRPIPE, user.Uid, STRDOT, user.PasswordCheckDelta, STRDOT, MakeFlags(user), MakeExt(user));
        }

        return body;
    }

    bool TSessionParser::ParseCreationTime(TSession& sess, TStringBuf view, TString& msg) {
        sess.Strtime_ = view;

        time_t ltime = 0;
        if (!TryIntFromString<10>(sess.Strtime_, ltime)) {
            msg.assign("Cookie with invalid timestamp was passed to ParseCookie");
            return false;
        }
        sess.Ts_ = ltime;

        return true;
    }

    bool TSessionParser::ParseSessionBlock(TSession& sess, TStringBuf view, TString& msg) {
        TStringBuf timeView = view.NextTok(DOT);
        TStringBuf ttlView = view.NextTok(DOT);
        TStringBuf defaultView = view.NextTok(DOT);
        TStringBuf authidView = view.NextTok(DOT);
        TStringBuf flagsView = view.NextTok(DOT);

        TSession::TSessionData& data = sess.Data_;

        // check for parts sanity
        if (
            timeView.size() < 10 ||   // strtime_
            ttlView.size() != 1 ||    // ttl_
            defaultView.empty() ||    // defIdx_
            authidView.size() < 16 || // authid_
            flagsView.empty()         // flags_
        ) {
            msg.assign("Cookie with bad field in per-session block was passed to ParseCookie");
            return false;
        }

        if (!ParseCreationTime(sess, timeView, msg)) {
            return false;
        }
        sess.Ttl_ = ttlView;

        long defIdx = 0;
        if (!TryIntFromString<16>(defaultView, defIdx)) {
            msg.assign("Cookie with bad default uid index was passed to ParseCookie");
            return false;
        }
        sess.Users_.SetDefIdx(defIdx);

        data.Authid.Str = authidView;
        if (!ParseAuthId(data.Authid)) {
            msg.assign("Cookie with bad authid was passed to ParseCookie");
            return false;
        }

        long flags = 0;
        if (!TryIntFromString<16>(flagsView, flags)) {
            msg.assign("Cookie with bad flags field was passed to ParseCookie");
            return false;
        }

        data.Safe = flags & sessSafe;
        data.Suspicious = flags & sessSuspicious;
        data.Stress = flags & sessStress;

        // parse known extension fields
        while (view) {
            TStringBuf entryView = view.NextTok(DOT);
            TStringBuf idView = entryView.NextTok(COLON);
            if (idView == SESS_USER_AGENT) {
                data.UserAgent = entryView;
            } else if (idView == SESS_LOGIN_ID) {
                data.LoginId = NUtils::Base64url2bin(entryView);
            } else if (idView == SESS_ENVIRONMENT) {
                data.Environment = GetEnvironmentType(entryView);
                if (!data.Environment) {
                    msg.assign("Cookie with incorrect environment field was passed to ParseCookie");
                    return false;
                }
            } else if (idView == SESS_EXT_USERIP) {
                data.ExtUserip = entryView;
            } else if (idView == SESS_EXT_AUTH_TS) {
                data.ExtAuthTs = entryView;
            } else if (idView == SESS_EXT_UPDATE_TS) {
                data.ExtUpdateTs = entryView;
            }
        }

        return true;
    }

    bool TSessionParser::ParseUserBlock(TSession& sess, TStringBuf view, TString& msg) {
        TStringBuf uidView = view.NextTok(DOT);
        TStringBuf pwdView = view.NextTok(DOT);
        TStringBuf flagsView = view.NextTok(DOT);

        TSession::TUserData& data = sess.Users_.AddUser();

        if (uidView.empty() || pwdView.empty() || flagsView.empty()) {
            msg.assign("Cookie with bad user block format was passed to ParseCookie");
            return false;
        }

        data.Uid = uidView;

        ui64 u = 0;
        if (!TryIntFromString<10>(uidView, u)) {
            msg.assign("Cookie with bad uid was passed to ParseCookie");
            return false;
        }

        long delta = -1;
        if (pwdView) {
            if (!TryIntFromString<10>(pwdView, delta)) {
                msg.assign("Cookie with bad password_check_delta was passed to ParseCookie");
                return false;
            }
        }
        data.PasswordCheckDelta = delta >= 0 ? delta : -1;

        long flags = 0;
        if (!TryIntFromString<16>(flagsView, flags)) {
            msg.assign("Cookie with bad user flags was passed to ParseCookie");
            return false;
        }

        data.Lite = flags & userLite;
        data.HavePassword = flags & userHavePwd;
        data.Staff = flags & userStaff;
        data.Betatester = flags & userBetatester;
        data.Glogouted = flags & userGlogouted;
        data.InternalAuth = flags & userInternalAuth;
        data.ExternalAuth = flags & userExternalAuth;
        data.ExtGlogouted = flags & userExtGlogouted;
        data.Scholar = flags & userScholar;

        // parse known extension fields
        while (view) {
            TStringBuf entryView = view.NextTok(DOT);
            TStringBuf idView = entryView.NextTok(COLON);
            if (idView == TStringBuf(USER_LANG)) {
                data.Lang = entryView;
            } else if (idView == TStringBuf(USER_SOCIALID)) {
                data.SocialId = entryView;
            } else if (idView == TStringBuf(USER_LOGIN_DELTA)) {
                long logindelta = 0;
                if (!TryIntFromString<10>(entryView, logindelta)) {
                    msg.assign("Cookie with bad user login delta was passed to ParseCookie");
                    return false;
                }
                data.LoginDelta = logindelta;
            }
        }

        return true;
    }

    bool TSessionParser::ParseUsers(TSession& sess, TStringBuf view, TString& msg) {
        // remove pre-added user
        sess.Users_.RemoveUser(0);

        // second, parse all users
        while (view) {
            TStringBuf block = view.NextTok(PIPELINE);
            if (!ParseUserBlock(sess, block, msg)) {
                // in case no users parsed successully add dummy user
                if (!sess.Users_.Count()) {
                    sess.Users_.AddUser();
                }
                if (sess.Users_.DefIdx() > sess.Users_.Count() - 1) { // some user failed to parse and default index is out of range
                    sess.Users_.SetDefIdx(0);                         // avoid getting out of range if asked for uid
                }
                return false;
            }
        }

        if (sess.Users_.Count() == 0) {
            msg.assign("Version 3 cookie with bad number of blocks was passed to ParseCookie");
            return false;
        }

        if (sess.Users_.DefIdx() > sess.Users_.Count() - 1) {
            msg.assign("Default user index out of range");
            sess.Users_.SetDefIdx(0); // avoid getting out of range if asked for uid
            return false;
        }

        return true;
    }

    bool TSessionParser::TryParseAsNoAuthCookie(TSession& sess, const TString& cookie) {
        if (!cookie.StartsWith(NOAUTH)) {
            return false;
        }

        sess.Status_ = TSession::NOAUTH;

        TString errMsg;
        if (!ParseCreationTime(sess, TStringBuf(cookie.cbegin() + NOAUTH.size() + 1, cookie.cend()), errMsg)) {
            sess.Status_ = TSession::NO_COOKIE;
            sess.Lasterr_ = NSessionCodes::BAD_COOKIE_BODY;
            TLog::Debug() << errMsg << ": <" << cookie << ">";
        }

        // we have no signature for this kind of cookies thus only sanity check
        // NOTE: since we throw away microseconds part of time we must allow
        // for discrepancy in times in now_ and ts_
        if (sess.Now_ + 1 < sess.Ts_) {
            sess.Status_ = TSession::NO_COOKIE;
            sess.Lasterr_ = NSessionCodes::BAD_COOKIE_TS;
            TLog::Debug() << "NOAUTH kind of cookie was passed with timestamp set to future: <" << cookie << ">";
        }

        return true;
    }

    TSession TSessionParser::ParseCookie(const TString& cookie, const TString& hostname) const {
        // it's empty and NO_COOKIE
        TSession sess(*this);

        // 17 is the size of noauth:<10-digits> cookie; any other cookie is longer
        if (cookie.size() < 17 || cookie.size() > 2048) {
            sess.Lasterr_ = NSessionCodes::BAD_COOKIE_SIZE;
            TLog::Debug() << "Cookie of improper size was passed to ParseCookie";
            return sess;
        }

        // get now
        sess.Now_ = time(nullptr);

        // look if we have noauth cookie
        if (TryParseAsNoAuthCookie(sess, cookie)) {
            return sess;
        }

        // get cookie format version
        TString::size_type seppos = cookie.find(':');

        if (seppos != 1 || cookie[0] != '3') {
            sess.Status_ = TSession::CANT_CHECK;
            sess.Lasterr_ = NSessionCodes::BAD_COOKIE_BODY;
            TLog::Debug() << "Cookie of unsupported version passed to ParseCookie: <" << cookie << ">";
            return sess;
        }
        sess.Version_ = 3;

        TString errMsg;

        // parse per-session data
        size_t pipelinePos = cookie.find(PIPELINE, seppos + 1);
        if (!ParseSessionBlock(sess, TStringBuf(cookie.cbegin() + seppos + 1, cookie.cbegin() + pipelinePos), errMsg)) {
            sess.Status_ = TSession::NO_COOKIE;
            sess.Lasterr_ = NSessionCodes::BAD_COOKIE_BODY;
            TLog::Debug() << errMsg << ": <" << cookie << ">";
            return sess;
        }

        // Validate will set lasterr_ by itself
        if (!sess.ValidateEnvironment(Environment_)) {
            TLog::Debug() << "Cookie from wrong environment passed to ParseCookie: <" << TStringBuf(cookie).RBefore(DOT) << ">";
            return sess;
        }

        // check signature
        TString::const_iterator tearoff;
        NSessionCodes::ESessionError res = SessSigner_.CheckSignature(cookie, sess.Ts_, sess.Domsuff_, tearoff);
        if (res != NSessionCodes::OK) {
            sess.Status_ = TSession::SIGN_BROKEN;
            sess.Lasterr_ = res;
            TLog::Debug() << "Cookie with bad signature passed to ParseCookie: <" << cookie << ">";
            return sess;
        }

        // finally, parse accounts data
        if (!ParseUsers(sess, TStringBuf(cookie.cbegin() + pipelinePos + 1, tearoff), errMsg)) {
            sess.Status_ = TSession::NO_COOKIE;
            sess.Lasterr_ = NSessionCodes::BAD_COOKIE_BODY;
            TLog::Debug() << errMsg << ": <" << cookie << ">";
            return sess;
        }

        // now we can validate cookie and set it's status_
        // Validate will set lasterr_ properly so dont touch it
        sess.Validate(hostname);

        return sess;
    }

    TSession TSessionParser::Create(TStringBuf uid, const TString& ttl, const TString& domsuff, time_t now) const {
        TSession res(*this);

        try {
            // Start with domain parts - these my cause an error
            //
            // Domain suffix - must be a configured one
            res.Domsuff_.assign(TSessionUtils::Dots2under(domsuff));
            const TKeyRing* keyring = SessSigner_.GetRingByName(res.Domsuff_);
            if (!keyring) {
                res.Lasterr_ = NSessionCodes::NO_DATA_KEYSPACE;
                throw yexception() << "Session::Create() error: unconfigured domain '"
                                   << domsuff << "')";
            }

            // Lifetime flag
            res.Ttl_ = ttl;

            TSession::TUserData& resuser = res.Users_.Get();

            // uid - must be numeric
            resuser.Uid = uid;
            ui64 u = 0;
            if (!TryIntFromString<10>(uid, u)) {
                throw yexception() << "Session::Create() error: invalid uid='" << uid << "'";
            }

            // Fine, can fill in everything else
            res.Ts_ = res.Now_ = now;
            res.Strtime_ = IntToString<10>(res.Ts_);
            res.Status_ = TSession::VALID;
            res.Lasterr_ = NSessionCodes::OK;
            res.Data_.Environment = Environment_;
        } catch (const std::exception& e) {
            TLog::Debug("%s", e.what());
        }

        return res;
    }

    TSessionErrorOr<TString> TSessionParser::MakeCookie(const TSession& src, TSession::ECategory category) const {
        TString body = MakeBodyV3(src, src.Strtime_, category);

        TSessionErrorOr<TString> signature = SessSigner_.MakeSigned(body, src.Domsuff_);

        body.append(signature.Value);
        return TSessionErrorOr<TString>{
            .Code = signature.Code,
            .Value = std::move(body),
        };
    }
}
