#pragma once

#include "session.h"
#include "session_error_or.h"
#include "session_errors.h"

#include <util/generic/strbuf.h>
#include <util/generic/string.h>

#include <map>
#include <memory>
#include <time.h>
#include <vector>

namespace NPassport::NAuth {
    class TSessionSigner;

    class TSessionParser {
    public:
        explicit TSessionParser(const TSessionSigner& sessSigner, EEnvironmentType env);

        const TSessionSigner& SessSigner() const {
            return SessSigner_;
        }

        // The only way to obtain session object is to call either one of
        // the following static methods.
        //
        // The first one parses session cookie into Session object
        TSession ParseCookie(const TString& cookie, const TString& hostname) const;

        // Creates cookie from scratch for the specified keyspace
        TSession Create(TStringBuf uid, const TString& ttl, const TString& domsuff, time_t now = time(nullptr)) const;

        // Makes session cookie string body and signature
        TSessionErrorOr<TString> MakeCookie(const TSession& src,
                                            TSession::ECategory category = TSession::ECategory::Actual) const;

        // Reference to the inside string storing error message.
        static const TString& ErrAsString(const NSessionCodes::ESessionError err);

        // parse authid field utils
        static time_t ParseAuthIdTs(TStringBuf authid_time);
        static bool ParseAuthId(TSession::TAuthid& authid);

    private:
        // =============== statics ================

        // make cookie and user flags and ext fields
        static TString MakeFlags(const TSession::TSessionData& data);
        static TString MakeFlags(const TSession::TUserData& data);
        static TString MakeExt(const TSession::TSessionData& data, TSession::ECategory category = TSession::ECategory::Actual);
        static TString MakeExt(const TSession::TUserData& data);
        // make v3 cookie body (session id or session info), without last pipeline (added in v3 MakeSigned)
        static TString MakeBodyV3(const TSession& src, const TString& strtime, TSession::ECategory category = TSession::ECategory::Actual);

        static bool ParseCreationTime(TSession& sess, TStringBuf view, TString& msg);
        static bool ParseSessionBlock(TSession& sess, TStringBuf view, TString& msg);
        static bool ParseUserBlock(TSession& sess, TStringBuf view, TString& msg);
        static bool ParseUsers(TSession& sess, TStringBuf view, TString& msg);

        static bool TryParseAsNoAuthCookie(TSession& sess, const TString& cookie);

    private:
        const TSessionSigner& SessSigner_;

        const EEnvironmentType Environment_;
    };

}
