#include "sessionsigner.h"

#include "keyring.h"
#include "random.h"
#include "sessionutils.h"

#include <passport/infra/libs/cpp/dbpool/handle.h>
#include <passport/infra/libs/cpp/juggler/status.h>
#include <passport/infra/libs/cpp/utils/regular_task.h>
#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/format.h>

#include <util/generic/string.h>

namespace NPassport::NAuth {
    TSessionSigner::TSessionSigner(NDbPool::TDbPool& defaultdb,
                                   const TKeyRingSettings& settings)
        : Dbp_(defaultdb)
        , KeyRingSettings_(settings)
    {
    }

    TSessionSigner::~TSessionSigner() = default;

    // TODO: PASSP-37512. keep here only `guard_passport`
    static const THashSet<TString> KEYRINGS_IN_DB = {
        "guard_mail",
        "guard_oauth",
        "guard_passport",
        "guard_sts",
        "guard_yasms",
        "guard_yasmstest",
    };

    void TSessionSigner::AddGuardSpace(const TString& id, const TString& spacename) {
        Y_ENSURE(!GuardSpaceNameById_.contains(id),
                 "Keyspace " << id << " is going to be initialized second time");
        Y_ENSURE(!GuardSpaceIdByName_.contains(spacename),
                 "Keyspace " << spacename << " is going to be initialized second time");

        GuardSpaceNameById_.emplace(id, spacename);
        GuardSpaceIdByName_.emplace(spacename, id);

        if (!KEYRINGS_IN_DB.contains(spacename)) {
            return;
        }

        try {
            TKeyRingPtr ring = std::make_shared<TKeyRing>(Dbp_, spacename, KeyRingSettings_);
            GuardSpaceByName_.insert(TKeyringMap::value_type(spacename, ring));
            GuardSpaceById_.insert(TKeyringMap::value_type(ring->GroupId(), ring));
            {
                std::lock_guard lock(ChainMutex_);
                RingChain_.push_back(ring);
            }
        } catch (const std::exception& e) {
            ythrow yexception() << "Failed to build keyring for spacename " << spacename << ": " << e.what();
        }
    }

    void TSessionSigner::CheckBeforeStart() const {
        Y_ENSURE(GetGuardRingByName("guard_passport"),
                 "SessionSigner misconfigured: missing 'guard_passport'");
    }

    NJuggler::TStatus TSessionSigner::GetJugglerStatus() const {
        for (const auto& [name, keyspace] : KeyringsByName_) {
            NJuggler::TStatus status = keyspace->GetJugglerStatus();
            if (status.StatusCode() != NJuggler::ECode::Ok) {
                return NJuggler::TStatus(status.StatusCode(), "At least one keyspace timed out: ", status.Message());
            }
        }
        return {};
    }

    NSessionCodes::ESessionError TSessionSigner::CheckSignature(const TString& data,
                                                                time_t createTime,
                                                                TString& reskspace,
                                                                TString::const_iterator& tearoff) const {
        // we look for
        // [.|]domsuf:keynum.rnd.sign
        // that was made as
        // .domsuf:keynum.rnd(nodot)key

        size_t dotpos = data.rfind('.');
        if (dotpos == TString::npos || (data.size() - dotpos) != 28 || dotpos < 6) {
            return NSessionCodes::BAD_SIGN;
        }
        // signature is (dotpos,end)
        TStringBuf sign(data.cbegin() + dotpos + 1, data.cend());
        TStringBuf body(data.cbegin(), data.cbegin() + dotpos);

        size_t nextpos = dotpos;
        dotpos = data.rfind('.', nextpos - 1);
        if (dotpos == TString::npos || nextpos - dotpos < 2 || dotpos < 4) {
            return NSessionCodes::BAD_SIGN;
        }
        // random is (dotpos,nextpos) - we aren't interested in it

        nextpos = dotpos;
        dotpos = data.rfind('|', nextpos - 1);
        if (dotpos == TString::npos || nextpos - dotpos < 2 || dotpos < 2) {
            return NSessionCodes::BAD_SIGN;
        }
        // ?domsuff:keynum is (dotpos,nextpos)
        tearoff = data.cbegin() + dotpos;

        size_t colpos = data.rfind(':', nextpos - 1);

        // domsuf must be explicitly specified (no default root keyspace)
        if (colpos == TString::npos || colpos < dotpos || nextpos - colpos < 2) {
            return NSessionCodes::BAD_SIGN;
        }

        // domsuff:keynum format
        TString kspace(data.cbegin() + dotpos + 1, data.cbegin() + colpos);
        TString keyid(data.cbegin() + colpos + 1, data.cbegin() + nextpos);

        TKeyRing* keyring = GetRingById(kspace);

        if (!keyring) {
            return NSessionCodes::NO_DATA_KEYSPACE;
        }

        // make it kspace name (it could be an id in cookie)
        reskspace = keyring->KSpace();

        // lookup the key by id and verify the hash value
        const TKeyWithGamma key = keyring->GetKeyById(keyid);
        if (key.Error()) {
            TLog::Debug() << "SessionSigner: failed to get key by id: " << keyid << ". " << key.Error();
            return time(nullptr) - createTime > 86400 * 100
                       ? NSessionCodes::TOO_OLD_COOKIE
                       : NSessionCodes::KEY_NOT_FOUND;
        }
        if (!key.IsCreateTimeValid(TInstant::Seconds(createTime))) {
            TLog::Debug() << "SessionSigner: key id " << keyid
                          << " is invalid by create time: " << TInstant::Seconds(createTime);
            return NSessionCodes::KEY_NOT_FOUND;
        }

        return TStringBuf(GetSignature(key, body)) == sign ? NSessionCodes::OK
                                                           : NSessionCodes::BAD_SIGN;
    }

    TSessionErrorOr<TString> TSessionSigner::MakeSigned(const TString& data,
                                                        const TString& domsuff) const {
        if (data.size() < 3 || data.size() > 2048) {
            return TSessionErrorOr<TString>{
                .Code = NSessionCodes::BAD_DATA_SIZE,
            };
        }

        const TKeyRing* keyring = GetRingByName(TSessionUtils::Dots2under(domsuff));
        if (!keyring) {
            return TSessionErrorOr<TString>{
                .Code = NSessionCodes::NO_DATA_KEYSPACE,
            };
        }

        const TKeyWithGamma key = keyring->GetKeyForSign();
        if (key.Error()) {
            TLog::Info("Request for newest key returned empty key for keyspace <%s>", domsuff.c_str());
            return TSessionErrorOr<TString>{
                .Code = NSessionCodes::KEYSPACE_EMPTY,
            };
        }

        const TString& kspace = keyring->GroupId();

        unsigned rnd = rand();
        char strrnd[20];
        snprintf(strrnd, 20, "%u", rnd % 1000000);

        // use base64 of 3 random bytes
        // TString strrnd = SessionUtils::bin2base64url((const char*)&rnd, 3);

        TString tocheck;
        tocheck.reserve(data.size() + 60);
        tocheck.assign(data);

        TString signature;
        signature.reserve(70);
        signature.assign("|");
        signature.append(kspace).append(":").append(key.Id().AsString());

        signature.append(".").append(strrnd);

        tocheck.append(signature);
        signature.append(".").append(GetSignature(key, tocheck));

        return TSessionErrorOr<TString>{
            .Code = NSessionCodes::OK,
            .Value = std::move(signature),
        };
    }

    TKeyRing* TSessionSigner::GetRingById(const TStringBuf keyspaceId) const {
        auto it = KeyringsById_.find(keyspaceId);
        return it == KeyringsById_.end() ? nullptr : it->second.get();
    }

    TKeyRing* TSessionSigner::GetRingByName(const TStringBuf keyspace) const {
        auto it = KeyringsByName_.find(keyspace);
        return it == KeyringsByName_.end() ? nullptr : it->second.get();
    }

    TKeyRing* TSessionSigner::GetGuardRingByName(const TStringBuf guardspace) const {
        auto it = GuardSpaceByName_.find(guardspace);
        return it == GuardSpaceByName_.end() ? nullptr : it->second.get();
    }

    TKeyRing* TSessionSigner::GetGuardRingById(const TStringBuf guardspaceId) const {
        auto it = GuardSpaceById_.find(guardspaceId);
        return it == GuardSpaceById_.end() ? nullptr : it->second.get();
    }

    static const TString EMPTY;

    const TString& TSessionSigner::GetGuardNameRingById(const TStringBuf guardspaceId) const {
        auto it = GuardSpaceNameById_.find(guardspaceId);
        return it == GuardSpaceNameById_.end() ? EMPTY : it->second;
    }

    const TString& TSessionSigner::GetGuardIdRingByName(const TStringBuf guardspaceName) const {
        auto it = GuardSpaceIdByName_.find(guardspaceName);
        return it == GuardSpaceIdByName_.end() ? EMPTY : it->second;
    }

    TKeyRing* TSessionSigner::TryToFindRingByName(const TStringBuf domain) const {
        TString keyspace = TSessionUtils::Dots2under(domain);

        do {
            auto it = KeyringsByName_.find(keyspace);
            if (it != KeyringsByName_.end()) {
                return it->second.get();
            }

            size_t pos = keyspace.find('_');
            if (pos == TString::npos) {
                break;
            }

            keyspace.erase(0, pos + 1);
        } while (!keyspace.empty());

        return nullptr;
    }

    bool TSessionSigner::CheckGuardSignature(const TStringBuf body, const TStringBuf signature,
                                             const TStringBuf keyspaceId, const TStringBuf keyspaceKeyId,
                                             const TStringBuf guardspaceId, const TStringBuf guardspaceKeyId) const {
        return CheckGuardSignatureInActualWay(body, signature, guardspaceKeyId) ||
               CheckGuardSignatureInLegacyWay(body, signature, keyspaceId, keyspaceKeyId, guardspaceId, guardspaceKeyId);
    }

    bool TSessionSigner::CheckGuardSignatureInLegacyWay(const TStringBuf body, const TStringBuf signature,
                                                        const TStringBuf keyspaceId, const TStringBuf keyspaceKeyId,
                                                        const TStringBuf guardspaceId, const TStringBuf guardspaceKeyId) const {
        const TKeyRing* keyring = GetRingById(keyspaceId);
        if (!keyring) {
            TLog::Warning() << "checkGuardSignature: couldn't find SessGuard keyspace id " << keyspaceId;
            return false;
        }

        const TRandom::EView keyView = TRandom::BodyHash;
        TKeyWithGamma key = keyring->GetKeyById(keyspaceKeyId, keyView);
        if (key.Error()) {
            TLog::Debug() << "checkGuardSignature: no key " << keyspaceKeyId << " in SessGuard keyspace " << keyspaceId;
            return false;
        }

        const TKeyRing* guardring = GetGuardRingById(guardspaceId);
        if (!guardring) {
            TLog::Warning() << "checkGuardSignature: couldn't find SessGuard guardspace id " << guardspaceId;
            return false;
        }

        const TRandomPtr guardRandom = guardring->GetRandomById(guardspaceKeyId);
        if (!guardRandom) {
            TLog::Debug() << "checkGuardSignature: no key " << guardspaceKeyId << " in SessGuard guardspace " << guardspaceId;
            return false;
        }

        key.Update(*guardRandom, keyView);

        TString expectedSignature = GetSignature(key, body);
        return signature == expectedSignature;
    }

    bool TSessionSigner::CheckGuardSignatureInActualWay(const TStringBuf body,
                                                        const TStringBuf signature,
                                                        const TStringBuf guardspaceKeyId) const {
        const TKeyRing* guardring = GetGuardRingByName("guard_passport");
        Y_ENSURE(guardring);

        TKeyWithGamma key = guardring->GetKeyById(guardspaceKeyId, TRandom::BodyHash);
        if (key.Error()) {
            TLog::Debug() << "checkGuardSignature: no key " << guardspaceKeyId << " in SessGuard keyspace guard_passport";
            return false;
        }

        TString expectedSignature = GetSignature(key, body);
        return signature == expectedSignature;
    }

    bool TSessionSigner::SignGuard(TString& guard, const TStringBuf keyspace, const TStringBuf guardspace) const {
        const TKeyRing* keyring = GetRingByName(keyspace);
        if (!keyring) {
            TLog::Warning() << "signGuard: couldn't find SessGuard keyspace " << keyspace;
            return false;
        }

        const TString& guardId = GetGuardIdRingByName(guardspace);
        if (guardId.empty()) {
            TLog::Debug() << "signGuard: SessGuard keyspace is unknown: " << keyspace;
            return false;
        }

        const TKeyRing* guardring = GetGuardRingByName("guard_passport");
        Y_ENSURE(guardring);

        TKeyWithGamma key = guardring->GetKeyForSign(TRandom::BodyHash);
        if (key.Error()) {
            TLog::Warning() << "signGuard: no key for sign for SessGuard keyspace " << keyspace;
            return false;
        }

        const TString random = NUtils::TCrypto::RandBytes(6);
        if (random.empty()) {
            TLog::Warning() << "signGuard: failed to generate random bytes";
            return false;
        }

        guard.reserve(guard.size() + 70); // reserve approximate signature size

        guard.append('.').append(keyring->GroupId());
        guard.append('.').append(guardId).append(':').append(key.Id().AsString());
        guard.append('.').append(NUtils::Bin2base64url(random));

        TString signature = GetSignature(key, guard);
        guard.append('.').append(signature);

        return true;
    }

    void TSessionSigner::AddKeyspace(const TString& kspacename, TDuration timeout) {
        TString domsuff = TSessionUtils::Dots2under(kspacename);
        Y_ENSURE(!KeyringsByName_.contains(domsuff),
                 "Keyspace " << domsuff << " is going to be initialized second time");

        try {
            // Insert same keyring to the map twice with two different keys:
            // 1) domainsuff (used almost always); 2) groupid (used for mobile
            // mail and probably others).
            //
            // ATTN: Rely on the fact that once created keyrings are never
            // cleaned up. If they were, we'd have to ensure we wouldn't delete
            // same keyring twice!
            //
            // Historical note: We use numerical group id instead of domainsuff
            // in v3 cookies to save cookie length.
            TKeyRingPtr krap = std::make_shared<TKeyRing>(Dbp_, domsuff, KeyRingSettings_, timeout);
            KeyringsByName_.insert(TKeyringMap::value_type(domsuff, krap));
            KeyringsById_.insert(TKeyringMap::value_type(krap->GroupId(), krap));
            {
                std::lock_guard lock(ChainMutex_);
                RingChain_.push_back(krap);
            }
        } catch (const std::exception& e) {
            ythrow yexception() << "Failed to build keyring for spacename '" << domsuff << "': " << e.what();
        }
    }

    TString TSessionSigner::GetSignature(const TKeyWithGamma& key, TStringBuf toSign) {
        return NUtils::Bin2base64url(NUtils::TCrypto::HmacSha1(key.Body(), toSign));
    }

    //
    // Worker thread syncing all registered keyspaces in the background
    //
    void TSessionSigner::Run() {
        std::lock_guard lock(ChainMutex_);
        for (const TKeyRingPtr& ch : RingChain_) {
            ch->TrySyncKeyRing();
        }
    }

}
