#pragma once

#include "keyring.h"
#include "session_error_or.h"
#include "session_errors.h"

#include <util/generic/hash.h>
#include <util/generic/strbuf.h>
#include <util/generic/string.h>

#include <memory>
#include <mutex>
#include <shared_mutex>
#include <vector>

namespace NPassport::NDbPool {
    class TDbPool;
}

namespace NPassport::NJuggler {
    class TStatus;
}

class TPasspAuthCoreSignerSuite;

namespace NPassport::NAuth {
    class TKeyRing;
    class TRandom;

    using TKeyRingPtr = std::shared_ptr<TKeyRing>;

    // static part of former Session: key management and signing functions
    class TSessionSigner {
    public:
        // There are two methods that may be used to specify which databases
        // used for which keyspace.
        // The first one sets default database. It will be used if no
        // database explicitly specified. This method is kind of initialization
        // entry and should be called only once. Further call to it will have
        // no effect.
        explicit TSessionSigner(NDbPool::TDbPool& defaultdb,
                                const TKeyRingSettings& settings = {});
        virtual ~TSessionSigner();

        // This thread-unsafe method adds keyspaces to list of ones to serve.
        // Only keyring added via this method or via SetDBForKeyspace will
        // be loaded. No keyrings will be loaded on demand.
        void AddKeyspace(const TString& kspacename, TDuration timeout = TDuration::Minutes(65));

        // Add space, return true on success
        void AddGuardSpace(const TString& id, const TString& spacename);

        void CheckBeforeStart() const;

        NJuggler::TStatus GetJugglerStatus() const;

        // This member checks arbitrary string for signature.
        // It tries to find at the end of the passed string formated tail, that
        // contains key id and signature hash. Look MakeSigned for more on
        // the format of this tail.
        // If signature check succeeded then SessionCodes::OK returned and:
        // resulting key space (domain suffix) is assigned to reskspace,
        // position at wich tail with signature starts (including dot) is
        // assigned to tearoff.
        NSessionCodes::ESessionError CheckSignature(const TString& data, time_t createTime,
                                                    TString& reskspace, TString::const_iterator& tearoff) const;
        // This member makes signature for passed data by the newest
        // key in specified keyspace and
        // puts resulting signature (including random and leading dot) into
        // returning value. One may append resulting signature to
        // data and pass then resulting string to CheckSignature.
        // The format of signature is following:
        // .[domainsuffix:]keyid.rnd.signhash
        TSessionErrorOr<TString> MakeSigned(const TString& data, const TString& domsuff) const;

        // discover suitable keyring for given keyspace
        // Call only after adding keyspace
        TKeyRing* GetRingById(const TStringBuf keyspaceId) const;
        TKeyRing* GetRingByName(const TStringBuf keyspace) const;

        // tries to find keyspace using suffixes of the domain
        TKeyRing* TryToFindRingByName(const TStringBuf domain) const;

        TKeyRing* GetGuardRingByName(const TStringBuf guardspace) const;
        TKeyRing* GetGuardRingById(const TStringBuf guardspaceId) const;
        const TString& GetGuardNameRingById(const TStringBuf guardspaceId) const;
        const TString& GetGuardIdRingByName(const TStringBuf guardspaceName) const;

        bool CheckGuardSignature(const TStringBuf body, const TStringBuf signature,
                                 const TStringBuf keyspaceId, const TStringBuf keyspaceKeyId,
                                 const TStringBuf guardspaceId, const TStringBuf guardspaceKeyId) const;
        bool SignGuard(TString& guard, const TStringBuf keyspace, const TStringBuf guardspace) const;

    protected:
        void Run();

    private:
        friend class ::TPasspAuthCoreSignerSuite;

        // We keep binding between keyspace (domain suffix) and keyring here
        using TKeyringMap = THashMap<TString, TKeyRingPtr>;
        TKeyringMap KeyringsByName_;
        TKeyringMap KeyringsById_;
        TKeyringMap GuardSpaceByName_;
        TKeyringMap GuardSpaceById_;
        THashMap<TString, TString> GuardSpaceNameById_;
        THashMap<TString, TString> GuardSpaceIdByName_;

        // here we actually store keyrings (for refreshing and deleting)
        using TKeyringChain = std::vector<TKeyRingPtr>;
        TKeyringChain RingChain_;

        // db pool to load rings from
        NDbPool::TDbPool& Dbp_;
        const TKeyRingSettings KeyRingSettings_;

        // compose the signature of given sequence
        static TString GetSignature(const TKeyWithGamma& key, TStringBuf toSign);

        bool CheckGuardSignatureInLegacyWay(const TStringBuf body, const TStringBuf signature,
                                            const TStringBuf keyspaceId, const TStringBuf keyspaceKeyId,
                                            const TStringBuf guardspaceId, const TStringBuf guardspaceKeyId) const;
        bool CheckGuardSignatureInActualWay(const TStringBuf body,
                                            const TStringBuf signature,
                                            const TStringBuf guardspaceKeyId) const;

        // mutex for modification of the binding of the keyspace and keyring
        mutable std::mutex ChainMutex_;
        // mutex for modification of global parameters (pool, root kspace, etc)
    };
}
