#include <passport/infra/libs/cpp/auth_core/gamma/gamma_keeper.h>

#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NAuth;

Y_UNIT_TEST_SUITE(GammaKeeper) {
    static std::shared_ptr<NSecretString::TSecretString> CreateBody(TStringBuf value) {
        return std::make_shared<NSecretString::TSecretString>(
            NUtils::TCrypto::Sha256(value));
    }

    static TGammaKeeperSettings CreateSettings() {
        return TGammaKeeperSettings{
            .Gammas = {
                {18, TGamma{
                         .TypesToCheck = {},
                         .Id = 18,
                         .Body = CreateBody("foooooooo"),
                     }},
                {19, TGamma{
                         .ValidUpperBound = TInstant::Seconds(900),
                         .TypesToCheck = {EEnitityType::OAuth},
                         .Id = 19,
                         .Body = CreateBody("baaaaaaar"),
                     }},
                {20, TGamma{
                         .TypesToCheck = {EEnitityType::Session},
                         .Id = 20,
                         .Body = CreateBody("baaaaaaar2"),
                     }},
            },
            .SignedTypes = {EEnitityType::Session},
            .SigningGamma = 18,
            .KeyspaceToType = {
                {"gammable_session", EEnitityType::Session},
                {"gammable_oauth", EEnitityType::OAuth},
            },
            .AllowToCheckWithoutGammaCreatedBefore = {
                {EEnitityType::OAuth, TInstant::Seconds(950)},
            },
        };
    }

    Y_UNIT_TEST(configuredSigning) {
        const TRandom random("123", "0123456789", 42);
        const TGammaKeeper keeper(CreateSettings());

        struct TCase {
            TString InKeyspace;
            TRandom::EView InView = TRandom::Body;
            TString OutBody;
            TCombinedKeyId OutId;
        };
        std::vector<TCase> cases = {
            TCase{
                .InKeyspace = "not_gammable",
                .InView = TRandom::Body,
                .OutBody = NUtils::Bin2hex("0123456789"),
                .OutId = 123,
            },
            TCase{
                .InKeyspace = "not_gammable",
                .InView = TRandom::BodyHash,
                .OutBody = "db24abe2d0a20fac957859eb755b3dbf695becdb9f5890e3a2a2411273ae50ad",
                .OutId = 123,
            },
            TCase{
                .InKeyspace = "gammable_oauth",
                .InView = TRandom::Body,
                .OutBody = NUtils::Bin2hex("0123456789"),
                .OutId = 123,
            },
            TCase{
                .InKeyspace = "gammable_session",
                .InView = TRandom::Body,
                .OutBody = "0d1db67e3ebe245b333719821c76870e311d0ebd6e06b1cf48906583395ced3e",
                .OutId = 180000123,
            },
            TCase{
                .InKeyspace = "gammable_session",
                .InView = TRandom::BinBody,
                .OutBody = "0d1db67e3ebe245b333719821c76870e311d0ebd6e06b1cf48906583395ced3e",
                .OutId = 180000123,
            },
        };

        for (const TCase& c : cases) {
            TKeyWithGamma key = keeper.GetForSign(c.InKeyspace, random, c.InView);
            UNIT_ASSERT_VALUES_EQUAL_C(key.Id(), c.OutId, c.OutId.AsNumber());
            UNIT_ASSERT_VALUES_EQUAL_C(NUtils::Bin2hex(key.Body()), c.OutBody, c.OutId.AsNumber());
        }
    }

    Y_UNIT_TEST(notConfiguredSigning) {
        const TRandom random("123", "0123456789", 42);
        TGammaKeeperSettings settings = CreateSettings();

        settings.SigningGamma = 42;
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TGammaKeeper(TGammaKeeperSettings{settings}),
            yexception,
            "missing singing gamma in settings: 42");

        settings.SigningGamma = 18;
        settings.SignedTypes.clear();

        const TGammaKeeper keeper(std::move(settings));

        for (const TString& c : {
                 "not_gammable",
                 "gammable_oauth",
                 "gammable_session",
             }) {
            TKeyWithGamma key = keeper.GetForSign(c, random, TRandom::Body);
            UNIT_ASSERT_VALUES_EQUAL(key.Id(), 123);
            UNIT_ASSERT_VALUES_EQUAL(key.Body(), "0123456789");
        }
    }

    Y_UNIT_TEST(checking) {
        const TRandom random("123", "0123456789", 42);
        const TGammaKeeper keeper(CreateSettings());

        struct TCase {
            TString CaseName;

            TString InKeyspace;
            TGammaId InId = 0;
            TRandom::EView InView = TRandom::Body;

            TCombinedKeyId OutId;
            TString OutBody;
            TInstant ValidCreateTime;
            std::optional<TInstant> InvalidCreateTime;

            TString Err;
        };

        std::vector<TCase> cases = {
            TCase{
                .CaseName = "not gammed entity for not gammable keyspace: use body",
                .InKeyspace = "not_gammable",
                .InId = 0,
                .InView = TRandom::Body,
                .OutId = 123,
                .OutBody = "30313233343536373839",
                .ValidCreateTime = TInstant::Max(),
            },
            TCase{
                .CaseName = "not gammed entity for not gammable keyspace: use binBody",
                .InKeyspace = "not_gammable",
                .InId = 0,
                .InView = TRandom::BinBody,
                .OutId = 123,
                .OutBody = "0123456789",
                .ValidCreateTime = TInstant::Max(),
            },
            TCase{
                .CaseName = "gammed but keyspace is not gammable",
                .InKeyspace = "not_gammable",
                .InId = 42,
                .Err = "keyspace 'not_gammable' is not configured for gamma=42",
            },
            TCase{
                .CaseName = "not gammed entity for gammable keyspace - limited create time: use body",
                .InKeyspace = "gammable_oauth",
                .InId = 0,
                .InView = TRandom::Body,
                .OutId = 123,
                .OutBody = "30313233343536373839",
                .ValidCreateTime = TInstant::Seconds(949),
                .InvalidCreateTime = TInstant::Seconds(950),
            },
            TCase{
                .CaseName = "not gammed entity for gammable keyspace - limited create time: use binBody",
                .InKeyspace = "gammable_oauth",
                .InId = 0,
                .InView = TRandom::BinBody,
                .OutId = 123,
                .OutBody = "0123456789",
                .ValidCreateTime = TInstant::Seconds(949),
                .InvalidCreateTime = TInstant::Seconds(950),
            },
            TCase{
                .CaseName = "not gammed entity for gammable keyspace - unlimited create time",
                .InKeyspace = "gammable_session",
                .InId = 0,
                .InView = TRandom::Body,
                .OutId = 123,
                .OutBody = "30313233343536373839",
                .ValidCreateTime = TInstant::Max(),
            },
            TCase{
                .CaseName = "gammed with unknown gammaid",
                .InKeyspace = "gammable_oauth",
                .InId = 42,
                .Err = "gamma=42 is unknown",
            },
            TCase{
                .CaseName = "gammed with unknown gammaid",
                .InKeyspace = "gammable_oauth",
                .InId = 42,
                .Err = "gamma=42 is unknown",
            },
            TCase{
                .CaseName = "gammed with gammaid forbidden for this type",
                .InKeyspace = "gammable_oauth",
                .InId = 18,
                .Err = "gamma=18 is not allowed to check keyspace 'gammable_oauth'",
            },
            TCase{
                .CaseName = "gammed for oauth",
                .InKeyspace = "gammable_oauth",
                .InId = 19,
                .InView = TRandom::Body,
                .OutId = 190000123,
                .OutBody = "f2282c618634e9109c86964f9be460539e8131c5a5a6499af648163898f9f5a0",
                .ValidCreateTime = TInstant::Seconds(899),
                .InvalidCreateTime = TInstant::Seconds(900),
            },
            TCase{
                .CaseName = "gammed for oauth: check random view to be unusable",
                .InKeyspace = "gammable_oauth",
                .InId = 19,
                .InView = TRandom::BinBody,
                .OutId = 190000123,
                .OutBody = "f2282c618634e9109c86964f9be460539e8131c5a5a6499af648163898f9f5a0",
                .ValidCreateTime = TInstant::Seconds(899),
                .InvalidCreateTime = TInstant::Seconds(900),
            },
            TCase{
                .CaseName = "gammed for session",
                .InKeyspace = "gammable_session",
                .InId = 20,
                .InView = TRandom::Body,
                .OutId = 200000123,
                .OutBody = "fb1dc327783f6de2ed9666b46fbd7cae0c931c84de9befe3c14d69e0c5f635e7",
                .ValidCreateTime = TInstant::Max(),
            },
        };

        for (const TCase& c : cases) {
            TKeyWithGamma key = keeper.GetForCheck(c.InKeyspace, random, c.InId, c.InView);
            if (c.Err) {
                UNIT_ASSERT_VALUES_EQUAL_C(c.Err, key.Error(), c.CaseName);
                continue;
            }

            UNIT_ASSERT_VALUES_EQUAL_C(key.Id(), c.OutId, c.CaseName);
            UNIT_ASSERT_VALUES_EQUAL_C(NUtils::Bin2hex(key.Body()), c.OutBody, c.CaseName);
            UNIT_ASSERT_C(key.IsCreateTimeValid(c.ValidCreateTime), c.CaseName);
            if (c.InvalidCreateTime) {
                UNIT_ASSERT_C(!key.IsCreateTimeValid(*c.InvalidCreateTime), c.CaseName);
            }
        }
    }
}
