#include <passport/infra/libs/cpp/auth_core/random.h>
#include <passport/infra/libs/cpp/auth_core/gamma/gamma.h>
#include <passport/infra/libs/cpp/auth_core/gamma/key_with_gamma.h>

#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NAuth;

Y_UNIT_TEST_SUITE(GammaGammedKey) {
    Y_UNIT_TEST(makeXor) {
        UNIT_ASSERT_VALUES_EQUAL(
            "aaeeaa",
            NUtils::Bin2hex(TKeyWithGamma::MakeXor(NUtils::Hex2bin("abcdef"), NUtils::Hex2bin("012345"))));

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            NUtils::Bin2hex(TKeyWithGamma::MakeXor("abcdef", "012")),
            yexception,
            "args have unequal size: left has 6, right has 3");
    }

    Y_UNIT_TEST(fromError) {
        TKeyWithGamma key = TKeyWithGamma::FromError("some error");

        UNIT_ASSERT_VALUES_EQUAL("some error", key.Error());
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            key.Body(),
            yexception,
            "incorrect using");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            key.Id(),
            yexception,
            "incorrect using");

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            key.SetValidUpperBound(TInstant()),
            yexception,
            "incorrect using");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            key.IsCreateTimeValid(TInstant()),
            yexception,
            "incorrect using");

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            key.Update(TRandom("123", "", 0), TRandom::Body),
            yexception,
            "incorrect using");
    }

    Y_UNIT_TEST(notGammedActually) {
        const TRandom random("123", "0123456789", 42);

        struct TCase {
            TRandom::EView View;
            TString Body;
        };
        std::vector<TCase> cases = {
            TCase{
                .View = TRandom::Body,
                .Body = "0123456789",
            },
            TCase{
                .View = TRandom::BinBody,
                .Body = NUtils::Hex2bin("0123456789"),
            },
            TCase{
                .View = TRandom::BodyHash,
                .Body = NUtils::Hex2bin("db24abe2d0a20fac957859eb755b3dbf695becdb9f5890e3a2a2411273ae50ad"),
            },
        };

        for (const TCase& c : cases) {
            TKeyWithGamma key(random, c.View);
            UNIT_ASSERT_VALUES_EQUAL_C(key.Id().AsNumber(), 123, c.View);
            UNIT_ASSERT_VALUES_EQUAL_C(key.Body(), c.Body, c.View);
        }
    }

    static std::shared_ptr<NSecretString::TSecretString> CreateBody(TStringBuf value) {
        return std::make_shared<NSecretString::TSecretString>(
            NUtils::TCrypto::Sha256(value));
    }

    Y_UNIT_TEST(gammed) {
        const TRandom random("123", "0123456789", 42);

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TKeyWithGamma(random,
                          TGamma{
                              .Body = std::make_shared<NSecretString::TSecretString>("foooooooo"),
                          }),
            yexception,
            "args have unequal size: left has 9, right has 32");

        TGamma gamma{
            .Id = 18,
            .Body = CreateBody("foooooooo"),
        };

        TKeyWithGamma gammed(random, gamma);
        UNIT_ASSERT_VALUES_EQUAL(gammed.Id().AsNumber(), 180000123);
        UNIT_ASSERT_VALUES_EQUAL(NUtils::Bin2hex(gammed.Body()), "57d524428c957acd36f5a09e93e9820b5b04252f55ccd03956dcfc3c9852fc6e");

        gamma.ValidUpperBound = TInstant::Seconds(789);
        {
            TKeyWithGamma gammed(random, gamma);
            UNIT_ASSERT(gammed.IsCreateTimeValid(TInstant::Seconds(0)));
            UNIT_ASSERT(!gammed.IsCreateTimeValid(TInstant::Seconds(789)));
            UNIT_ASSERT(!gammed.IsCreateTimeValid(TInstant::Seconds(1000)));

            gammed.SetValidUpperBound(TInstant::Seconds(900));
            UNIT_ASSERT(gammed.IsCreateTimeValid(TInstant::Seconds(0)));
            UNIT_ASSERT(gammed.IsCreateTimeValid(TInstant::Seconds(789)));
            UNIT_ASSERT(!gammed.IsCreateTimeValid(TInstant::Seconds(1000)));
        }
    }

    Y_UNIT_TEST(timeBound) {
        const TRandom random("123", "0123456789", 42);
        TGamma gamma{
            .Id = 18,
            .Body = CreateBody("foooooooo"),
        };

        {
            TKeyWithGamma gammed(random, gamma);
            UNIT_ASSERT(gammed.IsCreateTimeValid(TInstant::Seconds(0)));
            UNIT_ASSERT(gammed.IsCreateTimeValid(TInstant::Seconds(789)));
            UNIT_ASSERT(gammed.IsCreateTimeValid(TInstant::Seconds(1000)));
        }

        {
            TKeyWithGamma gammed(random, gamma);
            gammed.SetValidUpperBound(TInstant::Seconds(900));
            UNIT_ASSERT(gammed.IsCreateTimeValid(TInstant::Seconds(0)));
            UNIT_ASSERT(gammed.IsCreateTimeValid(TInstant::Seconds(789)));
            UNIT_ASSERT(!gammed.IsCreateTimeValid(TInstant::Seconds(1000)));
        }

        gamma.ValidUpperBound = TInstant::Seconds(789);
        {
            TKeyWithGamma gammed(random, gamma);
            UNIT_ASSERT(gammed.IsCreateTimeValid(TInstant::Seconds(0)));
            UNIT_ASSERT(!gammed.IsCreateTimeValid(TInstant::Seconds(789)));
            UNIT_ASSERT(!gammed.IsCreateTimeValid(TInstant::Seconds(1000)));

            gammed.SetValidUpperBound(TInstant::Seconds(900));
            UNIT_ASSERT(gammed.IsCreateTimeValid(TInstant::Seconds(0)));
            UNIT_ASSERT(gammed.IsCreateTimeValid(TInstant::Seconds(789)));
            UNIT_ASSERT(!gammed.IsCreateTimeValid(TInstant::Seconds(1000)));
        }
    }

    Y_UNIT_TEST(update) {
        const TRandom random("123", "0123456789", 42);
        const TRandom randomToUpdate("124", "abcdef", 43);

        TGamma gamma{
            .Id = 18,
            .Body = CreateBody("foooooooo"),
        };

        TKeyWithGamma gammed(random, gamma);
        UNIT_ASSERT_VALUES_EQUAL(NUtils::Bin2hex(gammed.Body()),
                                 "57d524428c957acd36f5a09e93e9820b5b04252f55ccd03956dcfc3c9852fc6e");

        for (TRandom::EView view : {TRandom::Body, TRandom::BinBody}) {
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                gammed.Update(randomToUpdate, view),
                yexception,
                "args have unequal size:");
        }

        UNIT_ASSERT_NO_EXCEPTION(gammed.Update(randomToUpdate, TRandom::BodyHash));
        UNIT_ASSERT_VALUES_EQUAL(NUtils::Bin2hex(gammed.Body()),
                                 "ce88878dd8c2fd1b69694dccf4a710e5da75ed532f8cd8931595107b4a44f5c9");
    }
}
