#include <passport/infra/libs/cpp/auth_core/keymap.h>

#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/file.h>

using namespace NPassport;
using namespace NPassport::NAuth;

Y_UNIT_TEST_SUITE(PasspAuthCoreKeyMap) {
    static const TString tmpFile = "tmp.keys";
    void WriteFile(TStringBuf body) {
        TFileOutput out(tmpFile);
        out << body;
    }

    Y_UNIT_TEST(createFromFile) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(TKeyMap::CreateFromFile("foo", 100500),
                                       yexception,
                                       R"(can't open "foo" with mode RdOnly|Seq)");

        WriteFile(R"(<keyset></keyset>)");
        UNIT_ASSERT_EXCEPTION_CONTAINS(TKeyMap::CreateFromFile(tmpFile, 100500),
                                       yexception,
                                       "No keys found in file: tmp.keys");
        WriteFile(R"(<keyset><key>k3E+</key></keyset>)");
        UNIT_ASSERT_EXCEPTION_CONTAINS(TKeyMap::CreateFromFile(tmpFile, 100500),
                                       NXml::TConfig::TMissingException,
                                       "nonexistent config param: /keyset/key[1]/@id");
        WriteFile(R"(<keyset><key id="100500">k3E-</key></keyset>)");
        UNIT_ASSERT_EXCEPTION_CONTAINS(TKeyMap::CreateFromFile(tmpFile, 100500),
                                       yexception,
                                       "Bad key id=100500: invalid base64");
        WriteFile(R"(<keyset><key id="100500">k3E+</key><key id="100500">q3E+</key></keyset>)");
        UNIT_ASSERT_EXCEPTION_CONTAINS(TKeyMap::CreateFromFile(tmpFile, 100500),
                                       yexception,
                                       "Key with id='100500' is duplicated in file: tmp.keys");

        WriteFile(R"(<keyset><key id="100500">k3E+</key><key id="100501">q3E+</key></keyset>)");
        UNIT_ASSERT_EXCEPTION_CONTAINS(TKeyMap::CreateFromFile(tmpFile, 1),
                                       yexception,
                                       "Default key with id='1' not found in keys");

        TKeyMap keys = TKeyMap::CreateFromFile(tmpFile, 100501);

        UNIT_ASSERT_VALUES_EQUAL(2, keys.GetSize());

        UNIT_ASSERT(keys.HasKey("100500"));
        UNIT_ASSERT(keys.HasKey("100501"));
        UNIT_ASSERT(!keys.HasKey("100502"));
        UNIT_ASSERT(!keys.HasKey(""));
        UNIT_ASSERT(keys.HasKey(100500));
        UNIT_ASSERT(keys.HasKey(100501));
        UNIT_ASSERT(!keys.HasKey(100502));
        UNIT_ASSERT(!keys.HasKey(0));

        UNIT_ASSERT_STRINGS_EQUAL("100501", keys.GetDefaultId());
        UNIT_ASSERT_VALUES_EQUAL(100501, keys.GetDefaultIdNum());
        UNIT_ASSERT_STRINGS_EQUAL("q3E+", NUtils::BinToBase64(keys.GetDefaultKey()));
        UNIT_ASSERT_STRINGS_EQUAL("k3E+", NUtils::BinToBase64(keys.GetKey("100500")));
        UNIT_ASSERT_STRINGS_EQUAL("q3E+", NUtils::BinToBase64(keys.GetKey("100501")));
        UNIT_ASSERT_STRINGS_EQUAL("k3E+", NUtils::BinToBase64(keys.GetKey(100500)));
        UNIT_ASSERT_STRINGS_EQUAL("q3E+", NUtils::BinToBase64(keys.GetKey(100501)));

        UNIT_ASSERT_EXCEPTION_CONTAINS(keys.GetKey("1"),
                                       yexception,
                                       "Key with id 1 not found");
        UNIT_ASSERT_EXCEPTION_CONTAINS(keys.GetKey(1),
                                       yexception,
                                       "Key with id 1 not found");
    }

    Y_UNIT_TEST(createFromMap) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(TKeyMap::CreateFromRawMap({}, 100500),
                                       yexception,
                                       "No keys found in map");
        UNIT_ASSERT_EXCEPTION_CONTAINS(TKeyMap::CreateFromRawMap({{1, ""}}, 2),
                                       yexception,
                                       "Default key with id='2' not found in keys");
        UNIT_ASSERT_EXCEPTION_CONTAINS(TKeyMap::CreateFromRawMap({{1, ""}}, 1),
                                       yexception,
                                       "Default key body is empty");

        TKeyMap keys = TKeyMap::CreateFromRawMap({{1, "123"}, {2, "456"}}, 2);
        UNIT_ASSERT_STRINGS_EQUAL("2", keys.GetDefaultId());
        UNIT_ASSERT_VALUES_EQUAL(2, keys.GetDefaultIdNum());
        UNIT_ASSERT_STRINGS_EQUAL("456", keys.GetDefaultKey());

        UNIT_ASSERT(keys.HasKey("1"));
        UNIT_ASSERT(keys.HasKey("2"));
        UNIT_ASSERT(!keys.HasKey("0"));
        UNIT_ASSERT(keys.HasKey(1));
        UNIT_ASSERT(keys.HasKey(2));
        UNIT_ASSERT(!keys.HasKey(0));

        UNIT_ASSERT_EXCEPTION_CONTAINS(keys.GetKey("3"),
                                       yexception,
                                       "Key with id 3 not found");
        UNIT_ASSERT_EXCEPTION_CONTAINS(keys.GetKey(3),
                                       yexception,
                                       "Key with id 3 not found");

        UNIT_ASSERT_STRINGS_EQUAL("123", keys.GetKey("1"));
        UNIT_ASSERT_STRINGS_EQUAL("123", keys.GetKey(1));
        UNIT_ASSERT_STRINGS_EQUAL("456", keys.GetKey("2"));
        UNIT_ASSERT_STRINGS_EQUAL("456", keys.GetKey(2));
    }

    Y_UNIT_TEST(defaultCtor) {
        TKeyMap keys;
        UNIT_ASSERT_VALUES_EQUAL(0, keys.GetSize());
        UNIT_ASSERT_VALUES_EQUAL(0, keys.GetDefaultIdNum());
        UNIT_ASSERT_VALUES_EQUAL("", keys.GetDefaultId());
        UNIT_ASSERT_VALUES_EQUAL("", keys.GetDefaultKey());
    }
}
