#include "common.h"

#include <passport/infra/libs/cpp/auth_core/keyring.h>
#include <passport/infra/libs/cpp/auth_core/oauth_token.h>
#include <passport/infra/libs/cpp/auth_core/oauth_token_parser.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NAuth;

Y_UNIT_TEST_SUITE(TPasspAuthCoreTokensSuite) {
    // NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
    std::unique_ptr<TOAuthTokenParser> P_;

    const TOAuthTokenParser& P() {
        if (!P_) {
            P_ = std::make_unique<TOAuthTokenParser>(
                TTestDbHolder::Pool(),
                TKeyRingSettings{.Signkeydepth = 2});
        }
        return *P_;
    }

    Y_UNIT_TEST(BadToken) {
        TOAuthToken t = P().ParseToken("");
        UNIT_ASSERT_EQUAL(TOAuthToken::NOT_INITED, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("token format broken", t.ErrMsg());

        t = P().ParseToken("blabla-bla");
        UNIT_ASSERT_EQUAL(TOAuthToken::NOT_INITED, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("token format broken", t.ErrMsg());

        t = P().ParseToken("2.123.12345.2222222222.1111.12345.ABCDEF.ABCABCABC.DEFDEF");
        UNIT_ASSERT_EQUAL(TOAuthToken::NOT_INITED, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("unknown token version", t.ErrMsg());

        t = P().ParseToken("1.-123.12345.2222222222.1111.12345.ABCDEF.ABCABCABC.DEFDEF");
        UNIT_ASSERT_EQUAL(TOAuthToken::NOT_INITED, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("malformed uid field", t.ErrMsg());

        t = P().ParseToken("1..12345.2222222222.1111.12345.ABCDEF.ABCABCABC.DEFDEF");
        UNIT_ASSERT_EQUAL(TOAuthToken::NOT_INITED, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("malformed uid field", t.ErrMsg());

        t = P().ParseToken("1.123..2222222222.1111111111222.12345.ABCDEF.ABCABCABC.DEFDEF");
        UNIT_ASSERT_EQUAL(TOAuthToken::NOT_INITED, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("token format broken", t.ErrMsg());

        t = P().ParseToken("1.123.12345..1111111111222.12345.ABCDEF.ABCABCABC.DEFDEF");
        UNIT_ASSERT_EQUAL(TOAuthToken::NOT_INITED, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("malformed expires field", t.ErrMsg());

        t = P().ParseToken("1.123.12345.2s222222s2.1111111111222.12345.ABCDEF.ABCABCABC.DEFDEF");
        UNIT_ASSERT_EQUAL(TOAuthToken::NOT_INITED, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("malformed expires field", t.ErrMsg());

        t = P().ParseToken("1.123.12345.2222222222..12345.ABCDEF.ABCABCABC.DEFDEF");
        UNIT_ASSERT_EQUAL(TOAuthToken::NOT_INITED, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("token format broken", t.ErrMsg());

        t = P().ParseToken("1.123.12345.2222222222.111111111.12345.ABCDEF.ABCABCABC.DEFDEF");
        UNIT_ASSERT_EQUAL(TOAuthToken::NOT_INITED, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("malformed token_id", t.ErrMsg());

        t = P().ParseToken("1.123.12345.2222222222.11111l1111222.12345.ABCDEF.ABCABCABC.DEFDEF");
        UNIT_ASSERT_EQUAL(TOAuthToken::NOT_INITED, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("malformed token_id", t.ErrMsg());

        t = P().ParseToken("1.123.12345.2222222222.111111111122222.12345.ABCDEF.ABCABCABC.DEFDEF");
        UNIT_ASSERT_EQUAL(TOAuthToken::NOT_INITED, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("malformed token_id", t.ErrMsg());

        t = P().ParseToken("1.123.12345.2222222222.1111122222333..ABCDEF.ABCABCABC.DEFDEF");
        UNIT_ASSERT_EQUAL(TOAuthToken::NOT_INITED, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("token format broken", t.ErrMsg());

        t = P().ParseToken("1.123.12345.2222222222.1111122222333.12345..ABCABCABC.DEFDEF");
        UNIT_ASSERT_EQUAL(TOAuthToken::NOT_INITED, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("token format broken", t.ErrMsg());

        t = P().ParseToken("1.123.12345.2222222222.1111122222333.12345.ABCDEF..DEFDEF");
        UNIT_ASSERT_EQUAL(TOAuthToken::NOT_INITED, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("token format broken", t.ErrMsg());

        t = P().ParseToken("1.123.12345.2222222222.1111122222333.12345.ABCDEF.ABCABCABC.");
        UNIT_ASSERT_EQUAL(TOAuthToken::NOT_INITED, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("token format broken", t.ErrMsg());

        t = P().ParseToken("1.123.12345.2222222222.1111122222333.12345.ABCDEF.ABCABCABC");
        UNIT_ASSERT_EQUAL(TOAuthToken::NOT_INITED, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("token format broken", t.ErrMsg());

        t = P().ParseToken("1.123.12345.1111111111.1111122222333.12345678.ABCDEF.ABCABCABC.DEFDEF");
        UNIT_ASSERT_EQUAL(TOAuthToken::EXPIRED, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("token expired", t.ErrMsg());

        t = P().ParseToken("1.123.12345.2222222222.1111122222333.123456.ABCDEF.ABCABCABC.DEFDEF");
        UNIT_ASSERT_EQUAL(TOAuthToken::SIGN_BROKEN, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("token key not found", t.ErrMsg());

        t = P().ParseToken("1.123.12345.2222222222.1111122222333.12345.ABCDEF.ABCABCABC.DEFDEF");
        UNIT_ASSERT_EQUAL(TOAuthToken::SIGN_BROKEN, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("failed to decrypt token data", t.ErrMsg());

        t = P().ParseToken("1.123.12345.2222222222.1111122222333.12345.ABCDEFABCDEFABCDEF.ABCABCABC.DEFDEF");
        UNIT_ASSERT_EQUAL(TOAuthToken::SIGN_BROKEN, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("failed to decrypt token data", t.ErrMsg());

        t = P().ParseToken("1.123.12345.2222222222.1111122222333.12345.ABCDEFABCDEFABCDEF.ABCABCABC.012301234567abcd0123_-");
        UNIT_ASSERT_EQUAL(TOAuthToken::SIGN_BROKEN, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("failed to decrypt token data", t.ErrMsg());
    }

    Y_UNIT_TEST(MakeThenParse) {
        TOAuthToken token = TOAuthTokenParser::CreateToken("1122334455", 1414141414, 2222223333, "777", "1122334455123");
        TOAuthToken::TScopeIds newscopes = {1, 5, 7, 99};
        token.SetScopes(newscopes);
        token.SetDeviceId("666");
        token.SetXtokenId("12345");
        token.SetXtokenShard(2);
        token.SetMeta("meta строка");
        token.SetLoginId("s:1544252683198:6wfvF8f5_WAHAQAAuAYCKg:7f");
        token.SetPaymentAuthContextId("abcd-12345");
        token.SetPaymentAuthScopeAddendum("{\"json\":[{\"key\":\"val\"}]}");

        TString err_msg;
        TString strtoken = P().MakeTokenStr(token, err_msg);
        UNIT_ASSERT_VALUES_EQUAL("", err_msg);
        UNIT_ASSERT_VALUES_EQUAL("1.1122334455.777.2222223333.1122334455123.12349.", strtoken.substr(0, 48));

        TOAuthToken t = P().ParseToken(strtoken);
        UNIT_ASSERT_EQUAL(TOAuthToken::VALID, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("", t.ErrMsg());
        UNIT_ASSERT_VALUES_EQUAL(1, t.Version());
        UNIT_ASSERT(!t.Expired());
        UNIT_ASSERT_VALUES_EQUAL(1414141414, t.CreateTime());
        UNIT_ASSERT_VALUES_EQUAL(2222223333, t.ExpireTime());
        UNIT_ASSERT_VALUES_EQUAL("1122334455", t.Uid());
        UNIT_ASSERT_VALUES_EQUAL("11223344551231122334455", t.TokenId());
        UNIT_ASSERT_VALUES_EQUAL("1122334455", t.TokenIdTime());
        UNIT_ASSERT_VALUES_EQUAL("777", t.ClientId());
        UNIT_ASSERT_VALUES_EQUAL("666", t.DeviceId());
        UNIT_ASSERT_VALUES_EQUAL("12345", t.XtokenId());
        UNIT_ASSERT_VALUES_EQUAL(2, t.XtokenShard());
        UNIT_ASSERT_VALUES_EQUAL("meta строка", t.Meta());
        UNIT_ASSERT_VALUES_EQUAL("s:1544252683198:6wfvF8f5_WAHAQAAuAYCKg:7f", t.LoginId());
        UNIT_ASSERT_VALUES_EQUAL("abcd-12345", t.PaymentAuthContextId());
        UNIT_ASSERT_VALUES_EQUAL("{\"json\":[{\"key\":\"val\"}]}", t.PaymentAuthScopeAddendum());
        TOAuthToken::TScopeIds scopes = t.Scopes();
        UNIT_ASSERT_VALUES_EQUAL(4, scopes.size());
        UNIT_ASSERT_UNEQUAL(scopes.end(), scopes.find(1));
        UNIT_ASSERT_UNEQUAL(scopes.end(), scopes.find(5));
        UNIT_ASSERT_UNEQUAL(scopes.end(), scopes.find(7));
        UNIT_ASSERT_UNEQUAL(scopes.end(), scopes.find(99));

        token = TOAuthTokenParser::CreateToken("", 0, 2222223333, "1111", "1231231231456");
        newscopes = {0};
        token.SetScopes(newscopes);

        strtoken = P().MakeTokenStr(token, err_msg);
        UNIT_ASSERT_VALUES_EQUAL("", err_msg);
        UNIT_ASSERT_VALUES_EQUAL("1.0.1111.2222223333.1231231231456.12349.", strtoken.substr(0, 40));

        t = P().ParseToken(strtoken);
        UNIT_ASSERT_EQUAL(TOAuthToken::VALID, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("", t.ErrMsg());
        UNIT_ASSERT_VALUES_EQUAL(1, t.Version());
        UNIT_ASSERT(!t.Expired());
        UNIT_ASSERT_VALUES_EQUAL(0, t.CreateTime());
        UNIT_ASSERT_VALUES_EQUAL(2222223333, t.ExpireTime());
        UNIT_ASSERT_VALUES_EQUAL("", t.Uid());
        UNIT_ASSERT_VALUES_EQUAL("12312312314560", t.TokenId());
        UNIT_ASSERT_VALUES_EQUAL("1231231231", t.TokenIdTime());
        UNIT_ASSERT_VALUES_EQUAL("1111", t.ClientId());
        UNIT_ASSERT_VALUES_EQUAL("", t.DeviceId());
        UNIT_ASSERT_VALUES_EQUAL("", t.XtokenId());
        UNIT_ASSERT_VALUES_EQUAL(0, t.XtokenShard());
        UNIT_ASSERT_VALUES_EQUAL("", t.Meta());
        UNIT_ASSERT_VALUES_EQUAL("", t.LoginId());
        scopes = t.Scopes();
        UNIT_ASSERT_VALUES_EQUAL(1, scopes.size());
        UNIT_ASSERT_UNEQUAL(scopes.end(), scopes.find(0));
    }

    Y_UNIT_TEST(ParseKnownTokens) {
        TOAuthToken t = P().ParseToken("1.1234567.123.2222222222.1111222233000.12345.3SfQ9LUjpoA0iizW.WbLDoUtYU8TLXdXtgBka3255rVLA__2EKL4uyVgZKuczgcFAeSftXl19Qb3l2nl-3gX7MUg7pQYgZZYaabQEBgGKIqwVcQjMFmSwpZG02zjp_g0dovg4-er-8AZdXVHaEKJyBI4WM3JuIhNveMpJSXoBDtkTFdA3JCflrcU-r-iAYVZhioOdw_Lzuw.D051K0_8C__sKEWZ0R7qcA");

        UNIT_ASSERT_EQUAL(TOAuthToken::VALID, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("", t.ErrMsg());
        UNIT_ASSERT_VALUES_EQUAL(1, t.Version());
        UNIT_ASSERT(!t.Expired());
        UNIT_ASSERT_VALUES_EQUAL(1541605076, t.CreateTime());
        UNIT_ASSERT_VALUES_EQUAL(2222222222, t.ExpireTime());
        UNIT_ASSERT_VALUES_EQUAL("1234567", t.Uid());
        UNIT_ASSERT_VALUES_EQUAL("11112222330001234567", t.TokenId());
        UNIT_ASSERT_VALUES_EQUAL("1111222233", t.TokenIdTime());
        UNIT_ASSERT_VALUES_EQUAL("123", t.ClientId());
        UNIT_ASSERT_VALUES_EQUAL("0123456789", t.DeviceId());
        UNIT_ASSERT_VALUES_EQUAL("1234", t.XtokenId());
        UNIT_ASSERT_VALUES_EQUAL(1, t.XtokenShard());
        UNIT_ASSERT_VALUES_EQUAL("меткая мета", t.Meta());
        UNIT_ASSERT_VALUES_EQUAL("t:some:custom.login_id string", t.LoginId());
        UNIT_ASSERT_VALUES_EQUAL("123-abc-7890", t.PaymentAuthContextId());
        UNIT_ASSERT_VALUES_EQUAL("{\"scopes\":[1,3,4,5],I'm broken..", t.PaymentAuthScopeAddendum());
        TOAuthToken::TScopeIds scopes = t.Scopes();
        UNIT_ASSERT_VALUES_EQUAL(3, scopes.size());
        UNIT_ASSERT_UNEQUAL(scopes.end(), scopes.find(7));
        UNIT_ASSERT_UNEQUAL(scopes.end(), scopes.find(8));
        UNIT_ASSERT_UNEQUAL(scopes.end(), scopes.find(9));

        t = P().ParseToken("1.0.123.2222222222.1111222233123.12350.Z3LEu3vHkG5nIszd.flNOToJs.X14rJAdMebwHb22GsPQ22w");
        UNIT_ASSERT_EQUAL(TOAuthToken::VALID, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("", t.ErrMsg());
        UNIT_ASSERT_VALUES_EQUAL(1, t.Version());
        UNIT_ASSERT(!t.Expired());
        UNIT_ASSERT_VALUES_EQUAL(1488558357, t.CreateTime());
        UNIT_ASSERT_VALUES_EQUAL(2222222222, t.ExpireTime());
        UNIT_ASSERT_VALUES_EQUAL("", t.Uid());
        UNIT_ASSERT_VALUES_EQUAL("11112222331230", t.TokenId());
        UNIT_ASSERT_VALUES_EQUAL("1111222233", t.TokenIdTime());
        UNIT_ASSERT_VALUES_EQUAL("123", t.ClientId());
        UNIT_ASSERT_VALUES_EQUAL("", t.DeviceId());
        UNIT_ASSERT_VALUES_EQUAL("", t.XtokenId());
        UNIT_ASSERT_VALUES_EQUAL(0, t.XtokenShard());
        UNIT_ASSERT_VALUES_EQUAL("", t.Meta());
        UNIT_ASSERT_VALUES_EQUAL("", t.LoginId());
        UNIT_ASSERT_VALUES_EQUAL("", t.PaymentAuthContextId());
        UNIT_ASSERT_VALUES_EQUAL("", t.PaymentAuthScopeAddendum());
        scopes = t.Scopes();
        UNIT_ASSERT_VALUES_EQUAL(0, scopes.size());

        t = P().ParseToken("1.0.123.1488555850.1111222233000.12350.3gFw3yckKafSyo3q.JMYlyR3WW4927I9Uvp8gppCY2DCY5NIHDNepmvX5quN6yJsaLg87DQYp4o57chvEhhFw9Q.0XEbe8rALY1twXUZC9uGM");
        UNIT_ASSERT_EQUAL(TOAuthToken::EXPIRED, t.Status());
        UNIT_ASSERT_VALUES_EQUAL("token expired", t.ErrMsg());
        UNIT_ASSERT_VALUES_EQUAL(1, t.Version());
        UNIT_ASSERT(t.Expired());
        UNIT_ASSERT_VALUES_EQUAL(0, t.CreateTime());
        UNIT_ASSERT_VALUES_EQUAL(1488555850, t.ExpireTime());
        UNIT_ASSERT_VALUES_EQUAL("", t.Uid());
        UNIT_ASSERT_VALUES_EQUAL("11112222330000", t.TokenId());
        UNIT_ASSERT_VALUES_EQUAL("1111222233", t.TokenIdTime());
        UNIT_ASSERT_VALUES_EQUAL("123", t.ClientId());
        UNIT_ASSERT_VALUES_EQUAL("", t.DeviceId());
        UNIT_ASSERT_VALUES_EQUAL("", t.XtokenId());
        UNIT_ASSERT_VALUES_EQUAL(0, t.XtokenShard());
        UNIT_ASSERT_VALUES_EQUAL("", t.Meta());
        UNIT_ASSERT_VALUES_EQUAL("", t.LoginId());
        UNIT_ASSERT_VALUES_EQUAL("", t.PaymentAuthContextId());
        UNIT_ASSERT_VALUES_EQUAL("", t.PaymentAuthScopeAddendum());
        scopes = t.Scopes();
        UNIT_ASSERT_VALUES_EQUAL(0, scopes.size());
    }
}
