#include <passport/infra/libs/cpp/auth_core/keymap.h>
#include <passport/infra/libs/cpp/auth_core/public_id.h>
#include <passport/infra/libs/cpp/auth_core/public_id_encryptor.h>

#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NAuth;

Y_UNIT_TEST_SUITE(PasspAuthCorePublicIdEncryptor) {
    static TPublicIdEncryptor CreateEncryptor() {
        return TPublicIdEncryptor(
            TKeyMap::CreateFromRawMap(
                {{1, NUtils::Hex2bin("d96c0686dd45d82326ef13b3140038d4672d36858b1c4bcbe62759ed578e389b")},
                 {2, NUtils::Hex2bin("7784a739e1d836297bf933005c486ec3cef9e0722adcc5ea81d83e46f8820516")}},
                1));
    }

    Y_UNIT_TEST(BadKeys) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TPublicIdEncryptor(TKeyMap::CreateFromRawMap({{1, "123"}, {5, "456"}}, 5)),
            yexception,
            "TPublicIdEncryptor: bad default key id (should be 0..3)");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TPublicIdEncryptor(TKeyMap::CreateFromRawMap({{1, "123"}, {2, "456"}}, 2)),
            yexception,
            "TPublicIdEncryptor: bad default key length (should be 32 bytes)");
    }

    Y_UNIT_TEST(PublicId) {
        TPublicId badId;
        UNIT_ASSERT_VALUES_EQUAL(TPublicId::BROKEN, badId.Status());
        UNIT_ASSERT_VALUES_EQUAL(0, badId.Uid());
        UNIT_ASSERT_VALUES_EQUAL(0, badId.Version());

        TPublicId id(1130000000010830ull, 3);
        UNIT_ASSERT_VALUES_EQUAL(TPublicId::VALID, id.Status());
        UNIT_ASSERT_VALUES_EQUAL(1130000000010830ull, id.Uid());
        UNIT_ASSERT_VALUES_EQUAL(3, id.Version());
    }

    Y_UNIT_TEST(Encrypt) {
        TPublicIdEncryptor enc = CreateEncryptor();
        TPublicId id(1130000000010830ull, 3);

        UNIT_ASSERT_STRINGS_EQUAL("k1965yccbjbeewh3mh89ezk689",
                                  enc.Encrypt(id));

        TPublicId id2(100500, 0);

        UNIT_ASSERT_STRINGS_EQUAL("fucka6ynnuv6pyk1b79v1pucgt",
                                  enc.Encrypt(id2));
    }

    static void CheckBadId(const TPublicIdEncryptor& enc, const TString& publicId) {
        TPublicId id = enc.Decrypt(publicId);

        UNIT_ASSERT_VALUES_EQUAL(TPublicId::BROKEN, id.Status());
        UNIT_ASSERT_VALUES_EQUAL(0, id.Uid());
        UNIT_ASSERT_VALUES_EQUAL(0, id.Version());
    }

    Y_UNIT_TEST(Decrypt) {
        TPublicIdEncryptor enc = CreateEncryptor();

        // parsing bad publicid
        CheckBadId(enc, "fucka6ynnuv6pyk1b79");
        CheckBadId(enc, "fucka6ynnuv6pyk1b79v1pucgt======");
        CheckBadId(enc, "fucka6ynnuv6pyk1b79v1pucgv");
        CheckBadId(enc, "fucka ynnuv6pyk b79v1pucgt");
        CheckBadId(enc, "fucka6ynnuv6pyk1b79vlpucgt");
        CheckBadId(enc, "fucka6ynnuv6pykIb79v1pucgt");
        CheckBadId(enc, "fucka6ynunv6pyk1b79v1pucgt");
        CheckBadId(enc, "e2yqx632wg543vctehrjp0xzyd"); // checksum mismatch inside

        // valid on key 1
        TPublicId id = enc.Decrypt("k1965yccbjbeewh3mh89ezk689");

        UNIT_ASSERT_VALUES_EQUAL(TPublicId::VALID, id.Status());
        UNIT_ASSERT_VALUES_EQUAL(1130000000010830ull, id.Uid());
        UNIT_ASSERT_VALUES_EQUAL(3, id.Version());

        id = enc.Decrypt("k1965yccbjBEEwh3mh89eZk689");

        UNIT_ASSERT_VALUES_EQUAL(TPublicId::VALID, id.Status());
        UNIT_ASSERT_VALUES_EQUAL(1130000000010830ull, id.Uid());
        UNIT_ASSERT_VALUES_EQUAL(3, id.Version());

        id = enc.Decrypt("fucka6ynnuv6pyk1b79v1pucgt");

        UNIT_ASSERT_VALUES_EQUAL(TPublicId::VALID, id.Status());
        UNIT_ASSERT_VALUES_EQUAL(100500, id.Uid());
        UNIT_ASSERT_VALUES_EQUAL(0, id.Version());

        id = enc.Decrypt("FUCKa6ynnuv6PYk1b79V1PUCgt");

        UNIT_ASSERT_VALUES_EQUAL(TPublicId::VALID, id.Status());
        UNIT_ASSERT_VALUES_EQUAL(100500, id.Uid());
        UNIT_ASSERT_VALUES_EQUAL(0, id.Version());

        // valid on key 2
        TPublicId id2 = enc.Decrypt("abxkn72zyd8bawxn9djrfcahv2");

        UNIT_ASSERT_VALUES_EQUAL(TPublicId::VALID, id2.Status());
        UNIT_ASSERT_VALUES_EQUAL(1130000000010830ull, id2.Uid());
        UNIT_ASSERT_VALUES_EQUAL(3, id2.Version());

        id2 = enc.Decrypt("Abxkn72zyd8bAWXn9djRFCahv2");

        UNIT_ASSERT_VALUES_EQUAL(TPublicId::VALID, id2.Status());
        UNIT_ASSERT_VALUES_EQUAL(1130000000010830ull, id2.Uid());
        UNIT_ASSERT_VALUES_EQUAL(3, id2.Version());

        id2 = enc.Decrypt("6468tbc715f4hfrp8vde54tr6j");

        UNIT_ASSERT_VALUES_EQUAL(TPublicId::VALID, id2.Status());
        UNIT_ASSERT_VALUES_EQUAL(100500, id2.Uid());
        UNIT_ASSERT_VALUES_EQUAL(0, id2.Version());

        id2 = enc.Decrypt("6468TBC715f4HFRP8vDe54TR6j");

        UNIT_ASSERT_VALUES_EQUAL(TPublicId::VALID, id2.Status());
        UNIT_ASSERT_VALUES_EQUAL(100500, id2.Uid());
        UNIT_ASSERT_VALUES_EQUAL(0, id2.Version());
    }
}
