#include "common.h"

#include <passport/infra/libs/cpp/auth_core/keyring.h>
#include <passport/infra/libs/cpp/auth_core/sessionsigner.h>

#include <passport/infra/libs/cpp/utils/log/file_logger.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/string.h>
#include <util/stream/file.h>
#include <util/system/fs.h>

using namespace NPassport;
using namespace NPassport::NAuth;
using namespace NPassport::NUtils;

static const TString logFileName("tmp_log_file");

Y_UNIT_TEST_SUITE(TPasspAuthCoreSigner) {
    Y_UNIT_TEST(GetRing) {
        TSessionSigner signer(TTestDbHolder::Pool());

        // no rings
        UNIT_ASSERT(!signer.GetRingById("yandex_ru"));
        UNIT_ASSERT(!signer.GetRingById("3"));
        UNIT_ASSERT(!signer.GetRingById("yandex_by"));
        UNIT_ASSERT(!signer.GetRingById("17"));
        UNIT_ASSERT(!signer.GetRingByName("yandex_ru"));
        UNIT_ASSERT(!signer.GetRingByName("3"));
        UNIT_ASSERT(!signer.GetRingByName("yandex_by"));
        UNIT_ASSERT(!signer.GetRingByName("17"));

        // added one
        UNIT_ASSERT_NO_EXCEPTION(signer.AddKeyspace("yandex_ru"));

        UNIT_ASSERT(signer.GetRingByName("yandex_ru"));
        UNIT_ASSERT(!signer.GetRingById("yandex_ru"));
        UNIT_ASSERT_VALUES_EQUAL(signer.GetRingByName("yandex_ru")->KSpace(), "yandex_ru");
        UNIT_ASSERT_VALUES_EQUAL(signer.GetRingByName("yandex_ru")->GroupId(), "3");
        UNIT_ASSERT(signer.GetRingById("3"));
        UNIT_ASSERT(!signer.GetRingByName("3"));
        UNIT_ASSERT_VALUES_EQUAL(signer.GetRingById("3")->KSpace(), "yandex_ru");
        UNIT_ASSERT_VALUES_EQUAL(signer.GetRingById("3")->GroupId(), "3");

        UNIT_ASSERT(!signer.GetRingById("yandex_by"));
        UNIT_ASSERT(!signer.GetRingById("17"));
        UNIT_ASSERT(!signer.GetRingByName("yandex_by"));
        UNIT_ASSERT(!signer.GetRingByName("17"));

        // added second ring
        UNIT_ASSERT_NO_EXCEPTION(signer.AddKeyspace("yandex_by"));

        UNIT_ASSERT(signer.GetRingByName("yandex_ru"));
        UNIT_ASSERT(!signer.GetRingById("yandex_ru"));
        UNIT_ASSERT_VALUES_EQUAL(signer.GetRingByName("yandex_ru")->KSpace(), "yandex_ru");
        UNIT_ASSERT_VALUES_EQUAL(signer.GetRingByName("yandex_ru")->GroupId(), "3");
        UNIT_ASSERT(signer.GetRingById("3"));
        UNIT_ASSERT(!signer.GetRingByName("3"));
        UNIT_ASSERT_VALUES_EQUAL(signer.GetRingById("3")->KSpace(), "yandex_ru");
        UNIT_ASSERT_VALUES_EQUAL(signer.GetRingById("3")->GroupId(), "3");

        UNIT_ASSERT(signer.GetRingByName("yandex_by"));
        UNIT_ASSERT(!signer.GetRingById("yandex_by"));
        UNIT_ASSERT_VALUES_EQUAL(signer.GetRingByName("yandex_by")->KSpace(), "yandex_by");
        UNIT_ASSERT_VALUES_EQUAL(signer.GetRingByName("yandex_by")->GroupId(), "17");
        UNIT_ASSERT(signer.GetRingById("17"));
        UNIT_ASSERT(!signer.GetRingByName("17"));
        UNIT_ASSERT_VALUES_EQUAL(signer.GetRingById("17")->KSpace(), "yandex_by");
        UNIT_ASSERT_VALUES_EQUAL(signer.GetRingById("17")->GroupId(), "17");

        // bad
        UNIT_ASSERT(!signer.GetRingById("yandex_ua"));
        UNIT_ASSERT(!signer.GetRingById("4"));
        UNIT_ASSERT(!signer.GetRingByName("yandex_ua"));
        UNIT_ASSERT(!signer.GetRingByName("4"));
    }

    Y_UNIT_TEST(TryToFindRing) {
        TSessionSigner signer(TTestDbHolder::Pool());

        UNIT_ASSERT(!signer.TryToFindRingByName("yandex.ru"));
        UNIT_ASSERT(!signer.TryToFindRingByName("mail.yandex.ru"));
        UNIT_ASSERT(!signer.TryToFindRingByName("beta.mail.yandex.ru"));
        UNIT_ASSERT(!signer.TryToFindRingByName("yandex.by"));
        UNIT_ASSERT(!signer.TryToFindRingByName("mail.yandex.by"));
        UNIT_ASSERT(!signer.TryToFindRingByName("beta.mail.yandex.by"));
        UNIT_ASSERT(!signer.GetRingByName("yandex_ru"));
        UNIT_ASSERT(!signer.GetRingByName("mail_yandex_ru"));

        UNIT_ASSERT_NO_EXCEPTION(signer.AddKeyspace("yandex_ru"));
        UNIT_ASSERT(signer.GetRingByName("yandex_ru"));
        UNIT_ASSERT(!signer.GetRingByName("mail_yandex_ru"));

        UNIT_ASSERT(signer.TryToFindRingByName("yandex.ru"));
        UNIT_ASSERT(signer.TryToFindRingByName("mail.yandex.ru"));
        UNIT_ASSERT(signer.TryToFindRingByName("beta.mail.yandex.ru"));
        UNIT_ASSERT(!signer.TryToFindRingByName("yandex.by"));
        UNIT_ASSERT(!signer.TryToFindRingByName("mail.yandex.by"));
        UNIT_ASSERT(!signer.TryToFindRingByName("beta.mail.yandex.by"));

        UNIT_ASSERT_NO_EXCEPTION(signer.AddKeyspace("yandex_by"));
        UNIT_ASSERT(signer.GetRingByName("yandex_by"));
        UNIT_ASSERT(!signer.GetRingByName("mail_yandex_by"));

        UNIT_ASSERT(signer.TryToFindRingByName("yandex.ru"));
        UNIT_ASSERT(signer.TryToFindRingByName("mail.yandex.ru"));
        UNIT_ASSERT(signer.TryToFindRingByName("beta.mail.yandex.ru"));
        UNIT_ASSERT(signer.TryToFindRingByName("yandex.by"));
        UNIT_ASSERT(signer.TryToFindRingByName("mail.yandex.by"));
        UNIT_ASSERT(signer.TryToFindRingByName("beta.mail.yandex.by"));

        UNIT_ASSERT(!signer.TryToFindRingByName(""));
        UNIT_ASSERT(!signer.TryToFindRingByName("."));
    }

    Y_UNIT_TEST(GetGuardRing) {
        TSessionSigner signer(TTestDbHolder::Pool());

        // no guard spaces
        UNIT_ASSERT(!signer.GetGuardRingByName("guard_passport"));
        UNIT_ASSERT(!signer.GetGuardRingById("1000"));
        UNIT_ASSERT(!signer.GetGuardRingByName("guard_oauth"));
        UNIT_ASSERT(!signer.GetGuardRingById("1001"));

        // added one
        UNIT_ASSERT_NO_EXCEPTION(signer.AddGuardSpace("1000", "guard_passport"));

        UNIT_ASSERT(signer.GetGuardRingByName("guard_passport"));
        UNIT_ASSERT_VALUES_EQUAL(signer.GetGuardRingByName("guard_passport")->KSpace(), "guard_passport");
        UNIT_ASSERT_VALUES_EQUAL(signer.GetGuardRingByName("guard_passport")->GroupId(), "1000");
        UNIT_ASSERT(signer.GetGuardRingById("1000"));
        UNIT_ASSERT_VALUES_EQUAL(signer.GetGuardRingById("1000")->KSpace(), "guard_passport");
        UNIT_ASSERT_VALUES_EQUAL(signer.GetGuardRingById("1000")->GroupId(), "1000");
        UNIT_ASSERT_VALUES_EQUAL(signer.GetGuardNameRingById("1000"), "guard_passport");

        UNIT_ASSERT(!signer.GetGuardRingByName("guard_oauth"));
        UNIT_ASSERT(!signer.GetGuardRingById("1001"));
        UNIT_ASSERT_VALUES_EQUAL(signer.GetGuardNameRingById("1001"), "");

        // added second
        UNIT_ASSERT_NO_EXCEPTION(signer.AddGuardSpace("1001", "guard_oauth"));

        UNIT_ASSERT(signer.GetGuardRingByName("guard_passport"));
        UNIT_ASSERT_VALUES_EQUAL(signer.GetGuardRingByName("guard_passport")->KSpace(), "guard_passport");
        UNIT_ASSERT_VALUES_EQUAL(signer.GetGuardRingByName("guard_passport")->GroupId(), "1000");
        UNIT_ASSERT(signer.GetGuardRingById("1000"));
        UNIT_ASSERT_VALUES_EQUAL(signer.GetGuardRingById("1000")->KSpace(), "guard_passport");
        UNIT_ASSERT_VALUES_EQUAL(signer.GetGuardRingById("1000")->GroupId(), "1000");
        UNIT_ASSERT_VALUES_EQUAL(signer.GetGuardNameRingById("1000"), "guard_passport");

        UNIT_ASSERT(signer.GetGuardRingByName("guard_oauth"));
        UNIT_ASSERT_VALUES_EQUAL(signer.GetGuardRingByName("guard_oauth")->KSpace(), "guard_oauth");
        UNIT_ASSERT_VALUES_EQUAL(signer.GetGuardRingByName("guard_oauth")->GroupId(), "1001");
        UNIT_ASSERT(signer.GetGuardRingById("1001"));
        UNIT_ASSERT_VALUES_EQUAL(signer.GetGuardRingById("1001")->KSpace(), "guard_oauth");
        UNIT_ASSERT_VALUES_EQUAL(signer.GetGuardRingById("1001")->GroupId(), "1001");
        UNIT_ASSERT_VALUES_EQUAL(signer.GetGuardNameRingById("1001"), "guard_oauth");

        // bad
        UNIT_ASSERT(!signer.GetGuardRingById("guard_passport"));
        UNIT_ASSERT(!signer.GetGuardRingByName("1000"));
        UNIT_ASSERT(!signer.GetGuardRingById("1002"));
        UNIT_ASSERT_VALUES_EQUAL(signer.GetGuardNameRingById("1002"), "");
    }

    Y_UNIT_TEST(CheckGuardSignature) {
        NFs::Remove(logFileName);
        std::unique_ptr log = std::make_unique<TFileLogger>(logFileName, "INFO", true);
        TLog::Init(std::move(log));

        TSessionSigner signer(TTestDbHolder::Pool());
        UNIT_ASSERT_NO_EXCEPTION(signer.AddKeyspace("yandex_ru"));
        UNIT_ASSERT_NO_EXCEPTION(signer.AddGuardSpace("1000", "guard_passport"));
        UNIT_ASSERT_NO_EXCEPTION(signer.AddGuardSpace("1001", "guard_oauth"));

        UNIT_ASSERT(!signer.CheckGuardSignature(
            "1.1559227777.1540542645970:AQAAfw:43..2:3335.100:1234.RZVJXoBj", "0c_GIvCWkfYwoZ1tvC033Sitv4U",
            "10", "32836", "1001", "23809"));

        UNIT_ASSERT(!signer.CheckGuardSignature(
            "1.1559227777.1540542645970:AQAAfw:43..2:3335.100:1234.RZVJXoBj", "0c_GIvCWkfYwoZ1tvC033Sitv4U",
            "3", "328", "1001", "23809"));

        UNIT_ASSERT(!signer.CheckGuardSignature(
            "1.1559227777.1540542645970:AQAAfw:43..2:3335.100:1234.RZVJXoBj", "0c_GIvCWkfYwoZ1tvC033Sitv4U",
            "3", "32836", "1000", "23809"));

        UNIT_ASSERT(!signer.CheckGuardSignature(
            "1.1559227777.1540542645970:AQAAfw:43..2:3335.100:1234.RZVJXoBj", "0c_GIvCWkfYwoZ1tvC033Sitv4U",
            "3", "32836", "guard_oauth", "23809"));

        UNIT_ASSERT(!signer.CheckGuardSignature(
            "1.1559227777.1540542645970:AQAAfw:43..2:3335.100:1234.RZVJXoBj", "0c_GIvCWkfYwoZ1tvC033Sitv4U",
            "3", "32836", "1001", "238"));

        UNIT_ASSERT(!signer.CheckGuardSignature(
            "1.1559227777.1540542645970:AQAAfw:43..2:3335.100:1234.RZVJXoBj", "bad_signature",
            "3", "32836", "1001", "23809"));

        UNIT_ASSERT(signer.CheckGuardSignature(
            "1.1559227777.1540542645970:AQAAfw:43..2:3335.100:1234.RZVJXoBj", "0c_GIvCWkfYwoZ1tvC033Sitv4U",
            "3", "32836", "1001", "23809"));

        UNIT_ASSERT(!signer.CheckGuardSignature(
            "1.1559227777.1540542645970:AQAAfw:43..2:3335.100:1234.RZVJXoBj", "0c_GIvCWkfYwoZ1tvC033Sitv4U",
            "yandex_ru", "32836", "1001", "23809"));

        {
            TLog::Reset();
            TFileInput log_file(logFileName);
            UNIT_ASSERT_STRINGS_EQUAL(
                "WARNING: checkGuardSignature: couldn't find SessGuard keyspace id 10\n"
                "WARNING: checkGuardSignature: couldn't find SessGuard guardspace id guard_oauth\n"
                "WARNING: checkGuardSignature: couldn't find SessGuard keyspace id yandex_ru\n",
                log_file.ReadAll());
        }
    }

    Y_UNIT_TEST(SignGuard) {
        NFs::Remove(logFileName);
        std::unique_ptr log = std::make_unique<TFileLogger>(logFileName, "INFO", true);
        TLog::Init(std::move(log));

        TSessionSigner signer(TTestDbHolder::Pool());
        UNIT_ASSERT_NO_EXCEPTION(signer.AddKeyspace("yandex_ua"));
        UNIT_ASSERT_NO_EXCEPTION(signer.AddGuardSpace("1000", "guard_passport"));
        UNIT_ASSERT_NO_EXCEPTION(signer.AddGuardSpace("1001", "guard_oauth"));

        TString guard("guard_body");

        UNIT_ASSERT(!signer.SignGuard(guard, "yandex_ru", "1000"));
        UNIT_ASSERT_STRINGS_EQUAL("guard_body", guard);

        UNIT_ASSERT(!signer.SignGuard(guard, "yandex_ua", "1000"));
        UNIT_ASSERT_STRINGS_EQUAL("guard_body", guard);

        UNIT_ASSERT(!signer.SignGuard(guard, "yandex_ua", "1001"));
        UNIT_ASSERT_STRINGS_EQUAL("guard_body", guard);

        UNIT_ASSERT(signer.SignGuard(guard, "yandex_ua", "guard_oauth"));
        UNIT_ASSERT_STRING_CONTAINS(guard, "guard_body.4.1001:7505.");

        UNIT_ASSERT(!signer.SignGuard(guard, "4", "guard_oauth"));

        {
            TLog::Reset();
            TFileInput log_file(logFileName);
            UNIT_ASSERT_STRINGS_EQUAL(
                "WARNING: signGuard: couldn't find SessGuard keyspace yandex_ru\n"
                "WARNING: signGuard: couldn't find SessGuard keyspace 4\n",
                log_file.ReadAll());
        }
    }
}
