#include "common.h"

#include <passport/infra/libs/cpp/auth_core/sessguard_parser.h>
#include <passport/infra/libs/cpp/auth_core/sessionsigner.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/string.h>

using namespace NPassport::NAuth;

Y_UNIT_TEST_SUITE(TPasspAuthCoreSessGuardParser) {
    class TSessGuardParserHolder {
    public:
        TSessionSigner Signer;
        TSessGuardParser Parser;

        TSessGuardParserHolder()
            : Signer(TTestDbHolder::Pool())
            , Parser(Signer)
        {
            UNIT_ASSERT_NO_EXCEPTION(Signer.AddKeyspace("yandex.ru"));
            UNIT_ASSERT_NO_EXCEPTION(Signer.AddKeyspace("yandex.ua"));

            UNIT_ASSERT_EXCEPTION_CONTAINS(Parser.AddGuardSpace("1001", "guard_oauth", {}), yexception, "No allowed keyspaces for guardspace guard_oauth");
            UNIT_ASSERT_NO_EXCEPTION(Parser.AddGuardSpace("1000", "guard_passport", {"passport", "passport-dev"}));
            UNIT_ASSERT_NO_EXCEPTION(Parser.AddGuardSpace("1001", "guard_oauth", {"oauth-test"}));
        }
    };

    static TSessGuardParser& P() {
        return Singleton<TSessGuardParserHolder>()->Parser;
    }

    Y_UNIT_TEST(GetGuardSpace) {
        UNIT_ASSERT_VALUES_EQUAL("", P().GetGuardSpace("", ""));

        UNIT_ASSERT_VALUES_EQUAL("", P().GetGuardSpace("passport-test.yandex.com", "yandex.com"));
        UNIT_ASSERT_VALUES_EQUAL("", P().GetGuardSpace("fakepassport.yandex.com", "yandex.com"));
        UNIT_ASSERT_VALUES_EQUAL("", P().GetGuardSpace(".yandex.com", "yandex.com"));
        UNIT_ASSERT_VALUES_EQUAL("", P().GetGuardSpace("yandex.com", "yandex.com"));
        UNIT_ASSERT_VALUES_EQUAL("", P().GetGuardSpace("ya.ru", "yandex.com"));
        UNIT_ASSERT_VALUES_EQUAL("", P().GetGuardSpace(".passport.yandex.ru", "ya.ru"));

        UNIT_ASSERT_VALUES_EQUAL("", P().GetGuardSpace("guard_passport", ""));
        UNIT_ASSERT_VALUES_EQUAL("", P().GetGuardSpace("guard_passport.yandex.ru", "yandex.ru"));
        UNIT_ASSERT_VALUES_EQUAL("", P().GetGuardSpace("guard_passport.yandex.ru", "yandex_ru"));
        UNIT_ASSERT_VALUES_EQUAL("guard_passport", P().GetGuardSpace("passport-dev.yandex.ua", "yandex.ua"));
        UNIT_ASSERT_VALUES_EQUAL("guard_passport", P().GetGuardSpace(".passport-dev.yandex.ua", "yandex.ua"));
        UNIT_ASSERT_VALUES_EQUAL("", P().GetGuardSpace(".passport-dev.yandex.ua", "yandex.ru"));
        UNIT_ASSERT_VALUES_EQUAL("", P().GetGuardSpace(".passport-dev.yandex.ua", "yandex_ua"));
        UNIT_ASSERT_VALUES_EQUAL("guard_passport", P().GetGuardSpace("www.passport-dev.yandex.com.tr", "yandex.com.tr"));
        UNIT_ASSERT_VALUES_EQUAL("guard_passport", P().GetGuardSpace("www.passport-dev.yandex.com.tr", ".yandex.com.tr"));
        UNIT_ASSERT_VALUES_EQUAL("", P().GetGuardSpace("www.passport-dev.yandex.com.tr", "somedomain.ru"));
        UNIT_ASSERT_VALUES_EQUAL("guard_passport", P().GetGuardSpace("passport.ya.ru", "ya.ru"));
        UNIT_ASSERT_VALUES_EQUAL("", P().GetGuardSpace("passport.ya.ru", "ya_ru"));
        UNIT_ASSERT_VALUES_EQUAL("", P().GetGuardSpace("passport.yandex.ru", ""));

        UNIT_ASSERT_VALUES_EQUAL("", P().GetGuardSpace("oauth.yandex.ru", "yandex.ru"));
        UNIT_ASSERT_VALUES_EQUAL("", P().GetGuardSpace("oauth-dev.yandex.ru", "yandex.ru"));
        UNIT_ASSERT_VALUES_EQUAL("guard_oauth", P().GetGuardSpace("aaa.bbb.www.oauth-test.yandex.com", "yandex.com"));
    }

    static void CheckGuard(const TSessGuard& g, TSessGuard::EStatus st, const TString& err,
                           const TString& authid = "", time_t ts = 0, const TString& keyspace = "", const TString& guardspace = "") {
        UNIT_ASSERT_VALUES_EQUAL(st, g.Status());
        UNIT_ASSERT_VALUES_EQUAL(err, g.ErrMsg());
        UNIT_ASSERT_VALUES_EQUAL(authid, g.AuthId());
        UNIT_ASSERT_VALUES_EQUAL(ts, g.CreateTime());
        UNIT_ASSERT_VALUES_EQUAL(1, g.Version());
        UNIT_ASSERT_VALUES_EQUAL(keyspace, g.Keyspace());
        UNIT_ASSERT_VALUES_EQUAL(guardspace, g.Guardspace());
    }

    Y_UNIT_TEST(ParseGuardErrors) {
        // format errors
        CheckGuard(P().ParseGuard("", ""),
                   TSessGuard::INVALID, "sessguard format broken");
        CheckGuard(P().ParseGuard("oops...", ""),
                   TSessGuard::INVALID, "sessguard format broken");
        CheckGuard(P().ParseGuard("1.2.3.4", ""),
                   TSessGuard::INVALID, "sessguard format broken");
        CheckGuard(P().ParseGuard(".12345.test:auth:id..3:32838.1000:7505.EnrGvH_Y.0wubAcQDGNIe3zgsoevlsDWnqFo", ""),
                   TSessGuard::INVALID, "unknown sessguard version");
        CheckGuard(P().ParseGuard("2.12345.test:auth:id..3:32838.1000:7505.EnrGvH_Y.0wubAcQDGNIe3zgsoevlsDWnqFo", ""),
                   TSessGuard::INVALID, "unknown sessguard version");
        CheckGuard(P().ParseGuard("1..test:auth:id..3:32838.1000:7505.EnrGvH_Y.0wubAcQDGNIe3zgsoevlsDWnqFo", ""),
                   TSessGuard::INVALID, "malformed ts field");
        CheckGuard(P().ParseGuard("1.x.test:auth:id..3:32838.1000:7505.EnrGvH_Y.0wubAcQDGNIe3zgsoevlsDWnqFo", ""),
                   TSessGuard::INVALID, "malformed ts field");
        CheckGuard(P().ParseGuard("1.12345...3:32838.1000:7505.EnrGvH_Y.0wubAcQDGNIe3zgsoevlsDWnqFo", ""),
                   TSessGuard::INVALID, "sessguard format broken");
        CheckGuard(P().ParseGuard("1.12345.test:auth:id.smth..1000:7505.EnrGvH_Y.0wubAcQDGNIe3zgsoevlsDWnqFo", ""),
                   TSessGuard::INVALID, "sessguard format broken");
        CheckGuard(P().ParseGuard("1.12345.test:auth:id..3:32838..EnrGvH_Y.0wubAcQDGNIe3zgsoevlsDWnqFo", ""),
                   TSessGuard::INVALID, "sessguard format broken");
        CheckGuard(P().ParseGuard("1.12345.test:auth:id..3:32838.1000:7505..0wubAcQDGNIe3zgsoevlsDWnqFo", ""),
                   TSessGuard::INVALID, "sessguard format broken");
        CheckGuard(P().ParseGuard("1.12345.test:auth:id..3:32838.1000:7505.EnrGvH_Y.", ""),
                   TSessGuard::INVALID, "sessguard format broken");

        // missing keys
        CheckGuard(P().ParseGuard("1.12345.test:auth:id..32838.1000:7505.EnrGvH_Y.0wubAcQDGNIe3zgsoevlsDWnqFo", ""),
                   TSessGuard::INVALID, "sessguard keyspace not found");
        CheckGuard(P().ParseGuard("1.12345.test:auth:id..5:32838.1000:7505.EnrGvH_Y.0wubAcQDGNIe3zgsoevlsDWnqFo", ""),
                   TSessGuard::INVALID, "sessguard keyspace not found");
        CheckGuard(P().ParseGuard("1.12345.test:auth:id..3.1000:7505.EnrGvH_Y.0wubAcQDGNIe3zgsoevlsDWnqFo", "passport.yandex.ru"),
                   TSessGuard::INVALID, "sessguard signature does not match");
        CheckGuard(P().ParseGuard("1.12345.test:auth:id..3:32838.7505.EnrGvH_Y.0wubAcQDGNIe3zgsoevlsDWnqFo", ""),
                   TSessGuard::INVALID, "sessguard guardspace not found");
        CheckGuard(P().ParseGuard("1.12345.test:auth:id..3:32838.100:7505.EnrGvH_Y.0wubAcQDGNIe3zgsoevlsDWnqFo", ""),
                   TSessGuard::INVALID, "sessguard guardspace not found");
        CheckGuard(P().ParseGuard("1.12345.test:auth:id..3:32838.1000.EnrGvH_Y.0wubAcQDGNIe3zgsoevlsDWnqFo", "passport-dev.yandex.ru"),
                   TSessGuard::INVALID, "sessguard signature does not match");
    }

    Y_UNIT_TEST(ParseGuardSign) {
        const TString validPasspRu = "1.12345.test:auth:id..3:32838.1000:7505.EnrGvH_Y.4BdwSawm1_VTb2JRU5yvAdTStww";
        const TString validPasspUa = "1.12345.test:auth:id..4:32768.1000:7505.At0TXtis.zH3TZ4J5vWWcTjTxk3X5XVYtPXE";
        const TString validOAuthRu = "1.12345.test:auth:id..3:32838.1001:23810.Ox-RaVFr.OHOVQwPgPi9r8rA7SjKsP_ZGxV8";
        const TString validOAuthUa = "1.12345.test:auth:id..4:32768.1001:23810.W_qMWZFI.hbO2J6a3kFuiqDlhMicenAY95Iw";

        // valid guard with host mismatch
        CheckGuard(P().ParseGuard(validPasspRu, "passport.yandex.com"), TSessGuard::INVALID, "sessguard keyspace does not match hostname");
        CheckGuard(P().ParseGuard(validPasspRu, "passport.ya.ru"), TSessGuard::INVALID, "sessguard keyspace does not match hostname");
        CheckGuard(P().ParseGuard(validPasspRu, "passport.heyyandex.ru"), TSessGuard::INVALID, "sessguard keyspace does not match hostname");
        CheckGuard(P().ParseGuard(validPasspRu, "passportyandex.ru"), TSessGuard::INVALID, "sessguard keyspace does not match hostname");
        CheckGuard(P().ParseGuard(validPasspRu, "yandex.ru"), TSessGuard::INVALID, "sessguard guardspace does not match hostname");
        CheckGuard(P().ParseGuard(validPasspRu, "passport-test.yandex.ru"), TSessGuard::INVALID, "sessguard guardspace does not match hostname");
        CheckGuard(P().ParseGuard(validPasspRu, "dev.yandex.ru"), TSessGuard::INVALID, "sessguard guardspace does not match hostname");
        CheckGuard(P().ParseGuard(validPasspRu, "nepassport.yandex.ru"), TSessGuard::INVALID, "sessguard guardspace does not match hostname");

        CheckGuard(P().ParseGuard(validPasspUa, "passport.yandex.com"), TSessGuard::INVALID, "sessguard keyspace does not match hostname");
        CheckGuard(P().ParseGuard(validPasspUa, "passport.ya.ru"), TSessGuard::INVALID, "sessguard keyspace does not match hostname");
        CheckGuard(P().ParseGuard(validPasspUa, "passport.heyyandex.ua"), TSessGuard::INVALID, "sessguard keyspace does not match hostname");
        CheckGuard(P().ParseGuard(validPasspUa, "passportyandex.ua"), TSessGuard::INVALID, "sessguard keyspace does not match hostname");
        CheckGuard(P().ParseGuard(validPasspUa, "yandex.ua"), TSessGuard::INVALID, "sessguard guardspace does not match hostname");
        CheckGuard(P().ParseGuard(validPasspUa, "passport-test.yandex.ua"), TSessGuard::INVALID, "sessguard guardspace does not match hostname");
        CheckGuard(P().ParseGuard(validPasspUa, "dev.yandex.ua"), TSessGuard::INVALID, "sessguard guardspace does not match hostname");
        CheckGuard(P().ParseGuard(validPasspUa, "nepassport.yandex.ua"), TSessGuard::INVALID, "sessguard guardspace does not match hostname");

        CheckGuard(P().ParseGuard(validOAuthRu, "yandex.ua"), TSessGuard::INVALID, "sessguard keyspace does not match hostname");
        CheckGuard(P().ParseGuard(validOAuthRu, "passport.yandex.ua"), TSessGuard::INVALID, "sessguard keyspace does not match hostname");
        CheckGuard(P().ParseGuard(validOAuthRu, "oauth-test.yandex.ua"), TSessGuard::INVALID, "sessguard keyspace does not match hostname");
        CheckGuard(P().ParseGuard(validOAuthRu, "yandex.ru"), TSessGuard::INVALID, "sessguard guardspace does not match hostname");
        CheckGuard(P().ParseGuard(validOAuthRu, "oauth.yandex.ru"), TSessGuard::INVALID, "sessguard guardspace does not match hostname");
        CheckGuard(P().ParseGuard(validOAuthRu, "no.oauth.yandex.ru"), TSessGuard::INVALID, "sessguard guardspace does not match hostname");
        CheckGuard(P().ParseGuard(validOAuthRu, "noauth-test.yandex.ru"), TSessGuard::INVALID, "sessguard guardspace does not match hostname");
        CheckGuard(P().ParseGuard(validOAuthRu, "dev-oauth-test.yandex.ru"), TSessGuard::INVALID, "sessguard guardspace does not match hostname");

        CheckGuard(P().ParseGuard(validOAuthUa, "yandex.ru"), TSessGuard::INVALID, "sessguard keyspace does not match hostname");
        CheckGuard(P().ParseGuard(validOAuthUa, "passport.yandex.ru"), TSessGuard::INVALID, "sessguard keyspace does not match hostname");
        CheckGuard(P().ParseGuard(validOAuthUa, "oauth-test.yandex.ru"), TSessGuard::INVALID, "sessguard keyspace does not match hostname");
        CheckGuard(P().ParseGuard(validOAuthUa, "yandex.ua"), TSessGuard::INVALID, "sessguard guardspace does not match hostname");
        CheckGuard(P().ParseGuard(validOAuthUa, "oauth.yandex.ua"), TSessGuard::INVALID, "sessguard guardspace does not match hostname");
        CheckGuard(P().ParseGuard(validOAuthUa, "no.oauth.yandex.ua"), TSessGuard::INVALID, "sessguard guardspace does not match hostname");
        CheckGuard(P().ParseGuard(validOAuthUa, "noauth-test.yandex.ua"), TSessGuard::INVALID, "sessguard guardspace does not match hostname");
        CheckGuard(P().ParseGuard(validOAuthUa, "dev-oauth-test.yandex.ua"), TSessGuard::INVALID, "sessguard guardspace does not match hostname");

        // host match
        CheckGuard(P().ParseGuard(validPasspRu, "passport.yandex.ru"), TSessGuard::VALID, "", "test:auth:id", 12345, "yandex_ru", "guard_passport");
        CheckGuard(P().ParseGuard(validPasspRu, "passport-dev.yandex.ru"), TSessGuard::VALID, "", "test:auth:id", 12345, "yandex_ru", "guard_passport");
        CheckGuard(P().ParseGuard(validPasspRu, "www.passport.yandex.ru"), TSessGuard::VALID, "", "test:auth:id", 12345, "yandex_ru", "guard_passport");
        CheckGuard(P().ParseGuard(validPasspRu, "inside.www.passport-dev.yandex.ru"), TSessGuard::VALID, "", "test:auth:id", 12345, "yandex_ru", "guard_passport");

        CheckGuard(P().ParseGuard(validPasspUa, "passport.yandex.ua"), TSessGuard::VALID, "", "test:auth:id", 12345, "yandex_ua", "guard_passport");
        CheckGuard(P().ParseGuard(validPasspUa, "sso.www.passport-dev.yandex.ua"), TSessGuard::VALID, "", "test:auth:id", 12345, "yandex_ua", "guard_passport");
        CheckGuard(P().ParseGuard(validPasspUa, ".passport-dev.yandex.ua"), TSessGuard::VALID, "", "test:auth:id", 12345, "yandex_ua", "guard_passport");

        CheckGuard(P().ParseGuard(validOAuthRu, "oauth-test.yandex.ru"), TSessGuard::VALID, "", "test:auth:id", 12345, "yandex_ru", "guard_oauth");
        CheckGuard(P().ParseGuard(validOAuthRu, "www.oauth-test.yandex.ru"), TSessGuard::VALID, "", "test:auth:id", 12345, "yandex_ru", "guard_oauth");
        CheckGuard(P().ParseGuard(validOAuthRu, ".inside.www.oauth-test.yandex.ru"), TSessGuard::VALID, "", "test:auth:id", 12345, "yandex_ru", "guard_oauth");

        CheckGuard(P().ParseGuard(validOAuthUa, "oauth-test.yandex.ua"), TSessGuard::VALID, "", "test:auth:id", 12345, "yandex_ua", "guard_oauth");
        CheckGuard(P().ParseGuard(validOAuthUa, "sso.www.oauth-test.yandex.ua"), TSessGuard::VALID, "", "test:auth:id", 12345, "yandex_ua", "guard_oauth");
        CheckGuard(P().ParseGuard(validOAuthUa, ".oauth-test.yandex.ua"), TSessGuard::VALID, "", "test:auth:id", 12345, "yandex_ua", "guard_oauth");
    }

    Y_UNIT_TEST(MakeGuardStr) {
        TSessGuard g("test:auth:id");
        TString domain;
        UNIT_ASSERT_VALUES_EQUAL("", P().MakeGuardStr(g, "", domain, 12345));
        UNIT_ASSERT(domain.empty());
        UNIT_ASSERT_VALUES_EQUAL("", P().MakeGuardStr(g, "yandex.com", domain, 12345));
        UNIT_ASSERT(domain.empty());
        UNIT_ASSERT_VALUES_EQUAL("", P().MakeGuardStr(g, "noyandex.ru", domain, 12345));
        UNIT_ASSERT(domain.empty());
        UNIT_ASSERT_VALUES_EQUAL("", P().MakeGuardStr(g, "passportyandex.ru", domain, 12345));
        UNIT_ASSERT(domain.empty());
        UNIT_ASSERT_VALUES_EQUAL("", P().MakeGuardStr(g, "www.yandex.ru", domain, 12345));
        UNIT_ASSERT(domain.empty());
        UNIT_ASSERT_VALUES_EQUAL("", P().MakeGuardStr(g, "oauth.yandex.ru", domain, 12345));
        UNIT_ASSERT(domain.empty());
        UNIT_ASSERT_VALUES_EQUAL("", P().MakeGuardStr(g, "noauth-test.yandex.ru", domain, 12345));
        UNIT_ASSERT(domain.empty());
        UNIT_ASSERT_VALUES_EQUAL("", P().MakeGuardStr(g, "passport_yandex_ru", domain, 12345));
        UNIT_ASSERT(domain.empty());
        UNIT_ASSERT_VALUES_EQUAL("", P().MakeGuardStr(g, "passport-test.yandex.ru", domain, 12345));
        UNIT_ASSERT(domain.empty());
        UNIT_ASSERT_VALUES_EQUAL("", P().MakeGuardStr(g, "passport.yandex.com", domain, 12345));
        UNIT_ASSERT(domain.empty());
        UNIT_ASSERT_VALUES_EQUAL("", P().MakeGuardStr(g, "1000.yandex.ru", domain, 12345));
        UNIT_ASSERT(domain.empty());

        UNIT_ASSERT_STRING_CONTAINS(P().MakeGuardStr(g, "passport-dev.yandex.ru", domain, 12345),
                                    "1.12345.test:auth:id..3.1000:7505.");
        UNIT_ASSERT_VALUES_EQUAL(".passport-dev.yandex.ru", domain);
        UNIT_ASSERT_STRING_CONTAINS(P().MakeGuardStr(g, "www.sso.passport.yandex.ua", domain, 12345),
                                    "1.12345.test:auth:id..4.1000:7505.");
        UNIT_ASSERT_VALUES_EQUAL(".passport.yandex.ua", domain);
        UNIT_ASSERT_STRING_CONTAINS(P().MakeGuardStr(g, ".oauth-test.yandex.ua", domain, 12345),
                                    "1.12345.test:auth:id..4.1001:7505.");
        UNIT_ASSERT_VALUES_EQUAL(".oauth-test.yandex.ua", domain);
    }
}
