#include "common.h"

#include <passport/infra/libs/cpp/auth_core/session.h>
#include <passport/infra/libs/cpp/auth_core/sessionparser.h>
#include <passport/infra/libs/cpp/auth_core/sessionsigner.h>

#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/dbpool/destination.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/string.h>
#include <util/string/subst.h>

#include <regex>

using namespace NPassport::NAuth;

Y_UNIT_TEST_SUITE(PasspAuthCoreSessionParser) {
    const TString Empty_ = "";
    const TString Uid_ = "12345";
    const TString Uid2_ = "54321";
    const TString Timestamp_ = IntToString<10>(time(nullptr));
    const TString Authid_ = "1391777454121:2130706433:127";
    const TString LoginId_ = "t:login_id:from_token";
    const TString ExtUserip_ = "6wfvF8f5_WAHAQAAuAYCKg";
    const TString ExtAuthts_ = "1500544995";
    const TString ExtUpdatets_ = "1515151515";
    const TString Socialid_ = "11222333";
    const long Logindelta_ = 543210l;
    const TString Defaultlang_ = "1";

    class TSessionParserHolder {
    public:
        TSessionSigner Signer;
        TSessionParser Parser;

        TSessionParser& operator()() {
            return Parser;
        }

        TSessionParserHolder()
            : Signer(TTestDbHolder::Pool())
            , Parser(Signer, EEnvironmentType::Testing)
        {
            UNIT_ASSERT_NO_EXCEPTION(Signer.AddKeyspace("yandex.ru"));
            UNIT_ASSERT_NO_EXCEPTION(Signer.AddKeyspace("yandex.ua"));
        }
    };

    // NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
    std::unique_ptr<TSessionParserHolder> P_;

    TSessionParser& P() {
        if (!P_) {
            P_ = std::make_unique<TSessionParserHolder>();
        }
        return P_->Parser;
    }

    Y_UNIT_TEST(OldTest) {
        // test cookie version 3: make one, compare with hand-made, parse and compare
        TSession sess = P().Create(Uid_, "3", "yandex.ua");
        sess.SetTime(Timestamp_);
        sess.TurnLite();
        sess.SetAuthId(Authid_);
        sess.SetStress(true);
        sess.SetLoginId(LoginId_);
        sess.SetExtUserIp(ExtUserip_);
        sess.SetExtAuthTs(ExtAuthts_);
        sess.SetExtUpdateTs(ExtUpdatets_);
        sess.SetExtGlogouted(true);
        sess.SetSafe(true);

        // add one more user
        UNIT_ASSERT_EQUAL(1u, sess.AddUser(Uid2_));
        sess.SetStaff(true, 1);
        sess.SetBetatester(true, 1);
        sess.SetGlogouted(true, 1);
        sess.SetInternalAuth(true, 1);
        sess.SetExternalAuth(true, 1);
        sess.SetScholar(true, 1);
        sess.SetPasswordCheckDelta(1000, 1);
        sess.SetHavePassword(true, 1);
        sess.SetLang("5", 1);
        sess.SetSocialId(Socialid_, 1);
        sess.SetLoginDelta(Logindelta_, 1);

        TString cookie = sess.AsString();

        UNIT_ASSERT_EQUAL(TSession::VALID, sess.IsValid());
        UNIT_ASSERT_EQUAL(2u, sess.UserCount());
        UNIT_ASSERT_EQUAL(0u, sess.DefaultIdx());
        UNIT_ASSERT(sess.HasExternalAuth());

        // compare with hand-made v3 cookie
        TString body("3:" + Timestamp_ + ".3.0." + Authid_ + ".101.1:dDpsb2dpbl9pZDpmcm9tX3Rva2Vu.2:2" +
                     ".100:" + ExtUserip_ + ".101:" + ExtAuthts_ + ".102:" + ExtUpdatets_ +
                     "|" + Uid_ + ".-1.801|" + Uid2_ + ".1000.e702.0:5.1:" + Socialid_ + ".2:543210|4:");

        UNIT_ASSERT_VALUES_EQUAL(body, cookie.substr(0, body.size()));

        TSession sess3 = P().ParseCookie(cookie, "yandex.ua");

        UNIT_ASSERT_EQUAL(TSession::VALID, sess3.IsValid());

        UNIT_ASSERT_EQUAL(3, sess3.Version());
        UNIT_ASSERT_EQUAL(2u, sess3.UserCount());
        UNIT_ASSERT_EQUAL(0u, sess3.DefaultIdx());
        UNIT_ASSERT(sess.HasExternalAuth());
        UNIT_ASSERT_EQUAL(sess.Ttl(), sess3.Ttl());
        UNIT_ASSERT_EQUAL(sess.LoginId(), sess3.LoginId());
        UNIT_ASSERT_EQUAL(sess.Environment(), sess3.Environment());
        UNIT_ASSERT_EQUAL(sess.ExtUserIp(), sess3.ExtUserIp());
        UNIT_ASSERT_EQUAL(sess.ExtAuthTs(), sess3.ExtAuthTs());
        UNIT_ASSERT_EQUAL(sess.ExtUpdateTs(), sess3.ExtUpdateTs());
        UNIT_ASSERT_EQUAL(sess.AuthId(), sess3.AuthId());
        UNIT_ASSERT_EQUAL(sess.IsStress(), sess3.IsStress());
        UNIT_ASSERT_EQUAL(sess.Safe(), sess3.Safe());
        UNIT_ASSERT_EQUAL(sess.IsStress(), sess3.IsStress());

        UNIT_ASSERT_EQUAL(sess.Uid(), sess3.Uid());
        UNIT_ASSERT_EQUAL(sess.IsLite(), sess3.IsLite());
        UNIT_ASSERT_EQUAL(sess.HavePassword(), sess3.HavePassword());
        UNIT_ASSERT_EQUAL(sess.PasswordCheckDelta(), sess3.PasswordCheckDelta());
        UNIT_ASSERT_EQUAL(sess.Lang(), sess3.Lang());
        UNIT_ASSERT_EQUAL(sess.IsBetatester(), sess3.IsBetatester());
        UNIT_ASSERT_EQUAL(sess.IsStaff(), sess3.IsStaff());
        UNIT_ASSERT_EQUAL(sess.IsGlogouted(), sess3.IsGlogouted());
        UNIT_ASSERT_EQUAL(sess.IsInternalAuth(), sess3.IsInternalAuth());
        UNIT_ASSERT_EQUAL(sess.IsExternalAuth(), sess3.IsExternalAuth());
        UNIT_ASSERT_EQUAL(sess.IsExtGlogouted(), sess3.IsExtGlogouted());
        UNIT_ASSERT_EQUAL(sess.IsScholar(), sess3.IsScholar());
        UNIT_ASSERT_EQUAL(sess.SocialId(), sess3.SocialId());
        UNIT_ASSERT_EQUAL(sess.LoginDelta(), sess3.LoginDelta());

        UNIT_ASSERT_EQUAL(sess.Uid(1), sess3.Uid(1));
        UNIT_ASSERT_EQUAL(sess.IsLite(1), sess3.IsLite(1));
        UNIT_ASSERT_EQUAL(sess.HavePassword(1), sess3.HavePassword(1));
        UNIT_ASSERT_EQUAL(sess.PasswordCheckDelta(1), sess3.PasswordCheckDelta(1));
        UNIT_ASSERT_EQUAL(sess.Lang(1), sess3.Lang(1));
        UNIT_ASSERT_EQUAL(sess.IsBetatester(1), sess3.IsBetatester(1));
        UNIT_ASSERT_EQUAL(sess.IsStaff(1), sess3.IsStaff(1));
        UNIT_ASSERT_EQUAL(sess.IsGlogouted(1), sess3.IsGlogouted(1));
        UNIT_ASSERT_EQUAL(sess.IsInternalAuth(1), sess3.IsInternalAuth(1));
        UNIT_ASSERT_EQUAL(sess.IsExternalAuth(1), sess3.IsExternalAuth(1));
        UNIT_ASSERT_EQUAL(sess.IsExtGlogouted(1), sess3.IsExtGlogouted(1));
        UNIT_ASSERT_EQUAL(sess.IsScholar(1), sess3.IsScholar(1));
        UNIT_ASSERT_EQUAL(sess.SocialId(1), sess3.SocialId(1));
        UNIT_ASSERT_EQUAL(sess.LoginDelta(1), sess3.LoginDelta(1));

        sess.RemoveUser(Uid_);

        UNIT_ASSERT_EQUAL(TSession::VALID, sess.IsValid());
        UNIT_ASSERT_EQUAL(1u, sess.UserCount());
        UNIT_ASSERT_EQUAL(0u, sess.DefaultIdx());

        UNIT_ASSERT_EQUAL(Uid2_, sess.Uid());
        UNIT_ASSERT_VALUES_EQUAL("3", sess.Ttl());
        UNIT_ASSERT_EQUAL(false, sess.IsLite());
        UNIT_ASSERT_EQUAL(true, sess.HavePassword());
        UNIT_ASSERT_EQUAL(1000, sess.PasswordCheckDelta());
        UNIT_ASSERT_VALUES_EQUAL("5", sess.Lang());
        UNIT_ASSERT_EQUAL(true, sess.IsBetatester());
        UNIT_ASSERT_EQUAL(true, sess.IsStaff());
        UNIT_ASSERT_EQUAL(true, sess.IsGlogouted());
        UNIT_ASSERT_EQUAL(true, sess.IsInternalAuth());
        UNIT_ASSERT_EQUAL(true, sess.IsExternalAuth());
        UNIT_ASSERT_EQUAL(false, sess.IsExtGlogouted());
        UNIT_ASSERT_EQUAL(Socialid_, sess.SocialId());
        UNIT_ASSERT_EQUAL(Logindelta_, sess.LoginDelta());
        UNIT_ASSERT_EQUAL(Authid_, sess.AuthId());
        UNIT_ASSERT(sess.HasExternalAuth());
    }

    Y_UNIT_TEST(WrongEnvironment) {
        TSession sess = P().Create(Uid_, "3", "yandex.ua");

        sess.SetTime(Timestamp_);
        sess.TurnLite();
        sess.SetAuthId(Authid_);
        sess.SetStress(true);
        sess.SetLoginId(LoginId_);
        sess.SetEnvironment(EEnvironmentType::Production);
        sess.SetSafe(true);

        const TString cookie = sess.AsString();

        TSession sess3 = P().ParseCookie(cookie, "yandex.ua");
        UNIT_ASSERT_VALUES_EQUAL(TSession::NO_COOKIE, sess3.IsValid());
        UNIT_ASSERT_VALUES_EQUAL(NSessionCodes::WRONG_ENVIRONMENT, sess3.LastErr());
        UNIT_ASSERT_VALUES_EQUAL("cookie was got in wrong environment: current environment is Testing, "
                                 "while cookie came from Production",
                                 sess3.LastErrAsString());
    }

    Y_UNIT_TEST(WithoutKey) {
        const time_t now = time(nullptr);

        TSession sess = P().Create(Uid_, "3", "yandex.ua");
        sess.SetTime(IntToString<10>(now - TInstant::Days(101).Seconds()));
        sess.TurnLite();
        sess.SetAuthId(Authid_);
        sess.SetStress(true);
        sess.SetExtUserIp(ExtUserip_);
        sess.SetExtAuthTs(ExtAuthts_);
        sess.SetExtUpdateTs(ExtUpdatets_);
        sess.SetExtGlogouted(true);

        const TString cookie = sess.AsString();

        TSession sess3 = P().ParseCookie(SubstGlobalCopy(cookie, "|4:32768.", "|4:123."), "yandex.ua");
        UNIT_ASSERT_VALUES_EQUAL(TSession::SIGN_BROKEN, sess3.IsValid());
        UNIT_ASSERT_VALUES_EQUAL(NSessionCodes::TOO_OLD_COOKIE, sess3.LastErr());
    }

    Y_UNIT_TEST(Statics) {
        UNIT_ASSERT_VALUES_EQUAL("no error", TSessionParser::ErrAsString(NSessionCodes::OK));
        UNIT_ASSERT_VALUES_EQUAL("cookie size is out of proper range",
                                 TSessionParser::ErrAsString(NSessionCodes::BAD_COOKIE_SIZE));
        UNIT_ASSERT_VALUES_EQUAL("cookie body doesn't corespond to proper format",
                                 TSessionParser::ErrAsString(NSessionCodes::BAD_COOKIE_BODY));
        UNIT_ASSERT_VALUES_EQUAL("the source for the object to copy is in bad state",
                                 TSessionParser::ErrAsString(NSessionCodes::INVALID_SOURCE));
        UNIT_ASSERT_VALUES_EQUAL("cookie timestamp field is corrupt",
                                 TSessionParser::ErrAsString(NSessionCodes::BAD_COOKIE_TS));
        UNIT_ASSERT_VALUES_EQUAL("hostname specified doesn't belong to cookie domain",
                                 TSessionParser::ErrAsString(NSessionCodes::HOST_DONT_MATCH));
        UNIT_ASSERT_VALUES_EQUAL("there is no data for given keyspace",
                                 TSessionParser::ErrAsString(NSessionCodes::NO_DATA_KEYSPACE));
        UNIT_ASSERT_VALUES_EQUAL("key with specified id isn't found",
                                 TSessionParser::ErrAsString(NSessionCodes::KEY_NOT_FOUND));
        UNIT_ASSERT_VALUES_EQUAL("no keys are defined in specified keyspace",
                                 TSessionParser::ErrAsString(NSessionCodes::KEYSPACE_EMPTY));
        UNIT_ASSERT_VALUES_EQUAL("signature has bad format or is broken",
                                 TSessionParser::ErrAsString(NSessionCodes::BAD_SIGN));
        UNIT_ASSERT_VALUES_EQUAL("size of input data is either too small or too big",
                                 TSessionParser::ErrAsString(NSessionCodes::BAD_DATA_SIZE));
        UNIT_ASSERT_VALUES_EQUAL("cookie is too old and cannot be checked",
                                 TSessionParser::ErrAsString(NSessionCodes::TOO_OLD_COOKIE));
        UNIT_ASSERT_VALUES_EQUAL("cookie was got in wrong environment (production/intranet_production/testing)",
                                 TSessionParser::ErrAsString(NSessionCodes::WRONG_ENVIRONMENT));
        UNIT_ASSERT_VALUES_EQUAL("unknown error", TSessionParser::ErrAsString(NSessionCodes::__LAST_DEPRECATED_CODE));
        // TODO test parseAuthId*
    }
}

using TStringMap = std::map<TString, TString>;
template <>
void Out<TStringMap>(IOutputStream& out, const TStringMap& value) {
    out << '{';
    bool first = true;
    for (const auto& pair : value) {
        if (!first) {
            out << ", ";
        } else {
            first = false;
        }
        out << '"' << pair.first << '"';
        out << ':';
        out << '"' << pair.second << '"';
    }
    out << '}';
}
