#include <passport/infra/libs/cpp/cache/cache.h>

#include <library/cpp/getopt/small/last_getopt.h>

#include <util/system/info.h>

#include <future>

using namespace NPassport::NUtils;
using namespace NPassport::NCache;

static const TDuration LIFETIME = TDuration::Seconds(1);
// NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
std::atomic_bool stoping = {false};
// NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
size_t threads;
// NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
int sleepSec;
// NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
TContextPtr<TString, TString> CTX_;

struct TPreparedData {
    void Prepare(i64 count) {
        Data.reserve(count / 600);

        while (count > 0) {
            TString key = GenString(100);
            TString value = GenString(500);

            count -= GetSizeOf(key) + GetSizeOf(value);

            Data.emplace_back(std::pair<TString, TString>{std::move(key), std::move(value)});
        }
    }

    std::vector<std::pair<TString, TString>> Data;

private:
    static TString GenString(size_t size) {
        TString res;
        res.reserve(size);
        for (size_t idx = 0; idx < size; ++idx) {
            res.push_back(rand());
        }
        return res;
    }
};

struct TRes {
    TDuration TimeSpent = TDuration::MicroSeconds(1);
    size_t Count = 0;

    constexpr TRes& operator+=(const TRes& o) noexcept {
        TimeSpent += o.TimeSpent;
        Count += o.Count;
        return *this;
    }
};

TRes Reader(TCachePtr<TString, TString> cache, const TPreparedData& data) {
    size_t count = 0;
    TInstant start = TInstant::Now();

    while (true) {
        for (const auto& pair : data.Data) {
            if (stoping.load(std::memory_order_relaxed)) {
                TInstant stop = TInstant::Now();
                return {stop - start, count};
            }

            ++count;
            volatile auto r = cache->Get(pair.first);
        }
    }
}

TRes Writer(TCachePtr<TString, TString> cache, const TPreparedData& data) {
    size_t count = 0;
    TInstant start = TInstant::Now();

    while (true) {
        for (const auto& pair : data.Data) {
            if (stoping.load(std::memory_order_relaxed)) {
                TInstant stop = TInstant::Now();
                return {stop - start, count};
            }

            ++count;
            cache->Put(TString(pair.first), TString(pair.second), LIFETIME);
        }
    }
}

void Run(size_t timeBucketCount, size_t keyBucketCount, double readersRate, const TPreparedData& data) {
    auto cache = CTX_->CreateCache(timeBucketCount, keyBucketCount, "userinfo");
    CTX_->StartCleaning(TDuration::Seconds(1));

    stoping = false;

    size_t readerCount = threads * readersRate;
    std::vector<std::future<TRes>> readers;
    for (size_t i = 0; i < readerCount; ++i) {
        readers.push_back(std::async(std::launch::async, [cache, &data]() {
            return Reader(cache, data);
        }));
    }
    std::vector<std::future<TRes>> writers;
    for (size_t i = 0; i < threads - readerCount; ++i) {
        writers.push_back(std::async(std::launch::async, [cache, &data]() {
            return Writer(cache, data);
        }));
    }
    Sleep(TDuration::Seconds(sleepSec));
    stoping = true;

    TRes totalRead;
    for (auto& a : readers) {
        totalRead += a.get();
    }
    TRes totalWritten;
    for (auto& a : writers) {
        totalWritten += a.get();
    }

    auto prRes = [](const TRes& res, size_t thr) {
        return size_t(1000000. * res.Count * thr / res.TimeSpent.MicroSeconds());
    };

    static int idx = 0;
    Cout << "idx=" << idx++
         << ". time_buckets=" << timeBucketCount
         << ". key_buckets=" << keyBucketCount
         << ". readers=" << readerCount
         << ". read_per_sec=" << prRes(totalRead, readerCount)
         << ". total_reads=" << totalRead.Count / 1000 << "k"
         << ". writers=" << threads - readerCount
         << ". written_per_sec=" << prRes(totalWritten, threads - readerCount)
         << ". total_writes=" << totalWritten.Count / 1000 << "k"
         << Endl;
}

// NOLINTNEXTLINE(bugprone-exception-escape)
int main(int argc, char** argv) {
    size_t totalSizeLimit;

    NLastGetopt::TOpts opts;
    opts.AddHelpOption();
    opts.AddLongOption("total_size_limit", "in Gb").StoreResult(&totalSizeLimit).DefaultValue(1);
    opts.AddLongOption("sleep", "in seconds").StoreResult(&sleepSec).DefaultValue(10);
    opts.AddCharOption('j', "threads").StoreResult(&threads).DefaultValue(NSystemInfo::NumberOfCpus());
    NLastGetopt::TOptsParseResult optRes(&opts, argc, argv);

    totalSizeLimit *= 1024 * 1024 * 1024;

    TPreparedData data;
    data.Prepare(1.5 * totalSizeLimit);

    for (double readers : {0., 0.25, 0.5, .75, 1.}) {
        for (size_t kBucketCount : {16, 32, 64, 128, 256, 512, 1024, 2048}) {
            for (size_t tBucketCount : {1, 2, 3, 4}) {
                CTX_ = std::make_shared<TContext<TString, TString>>(totalSizeLimit, "some_uni_name");
                Run(tBucketCount, kBucketCount, readers, data);
                CTX_.reset();
            }
        }
    }

    return 0;
}
