#pragma once

#include "cyclic_idx.h"
#include "sizeof.h"
#include "time_based_bucket.h"

#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/regular_task.h>

#include <util/generic/yexception.h>

#include <atomic>
#include <memory>
#include <optional>
#include <vector>

namespace NPassport::NCache {
    template <typename Key, typename Value>
    class TContext;

    template <typename Key, typename Value>
    using TContextPtr = std::shared_ptr<TContext<Key, Value>>;

    template <typename Key, typename Value>
    class TCache {
    public:
        TCache(TContextPtr<Key, Value> ctx,
               size_t timeBucketCount,
               size_t keyBucketCount,
               const TString& unistatName)
            : Ctx_(std::move(ctx))
            , WritingIdx_(0)
            , UnistatGetSucceed_("cache." + unistatName + ".get_succeed")
            , UnistatGetFailed_("cache." + unistatName + ".get_failed")
            , UnistatPutFailed_("cache." + unistatName + ".put_failed")
        {
            Y_ENSURE(timeBucketCount > 0, "bucket count must be > 0"); // recomended: timeBucketCount == 3
            Y_ENSURE(Ctx_);

            Buckets_.reserve(timeBucketCount);
            for (size_t idx = 0; idx < timeBucketCount; ++idx) {
                Buckets_.push_back(std::make_unique<TTimeBasedBucket<Key, Value>>(keyBucketCount));
            }
        }

        std::optional<Value> Get(const Key& key, TInstant now = TInstant::Now()) const {
            const TCyclicIdx writingIdx = GetWritingIdx();
            TCyclicIdx readingIdx = writingIdx;

            // everything except writingIdx
            while (readingIdx.Backward()) {
                std::optional<Value> res = Buckets_[readingIdx.GetValue()]->Get(key, now);
                if (res) {
                    ++UnistatGetSucceed_;
                    return res;
                }
            }

            std::optional<Value> res = Buckets_[writingIdx.GetValue()]->Get(key, now);
            if (res) {
                ++UnistatGetSucceed_;
            } else {
                ++UnistatGetFailed_;
            }
            return res;
        }

        size_t TryClean(TInstant now) {
            TCyclicIdx cleaningIdx = GetWritingIdx();
            cleaningIdx.Forward();
            const size_t idx = cleaningIdx.GetValue();

            std::optional<size_t> res = Buckets_[idx]->TryClean(now);
            if (res) {
                // Move writing idx only if bucket was cleaned
                WritingIdx_.store(idx, std::memory_order_release);
                return *res;
            }
            return 0;
        }

        void Put(Key&& key, Value&& value, TDuration lifeTime, TInstant now = TInstant::Now());

        void AddUnistat(NUnistat::TBuilder& builder) {
            builder.Add(UnistatGetSucceed_);
            builder.Add(UnistatGetFailed_);
            builder.Add(UnistatPutFailed_);
        }

    public: // for tests
        TCyclicIdx GetWritingIdx() const {
            return TCyclicIdx(WritingIdx_.load(std::memory_order_relaxed), Buckets_.size());
        }

    private:
        TContextPtr<Key, Value> Ctx_;
        std::atomic<ui64> WritingIdx_;

        mutable NUnistat::TSignalDiff<> UnistatGetSucceed_;
        mutable NUnistat::TSignalDiff<> UnistatGetFailed_;
        NUnistat::TSignalDiff<> UnistatPutFailed_;

        /**
         *        1 2 3
         *        | | |
         * |x|...|x|x|x|...|x|
         *
         * Positions:
         * 1 - start reading here and move backwards + read from writing position
         * 2 - write here
         * 3 - clean here and move forwards
         */
        std::vector<TTimeBasedBucketPtr<Key, Value>> Buckets_;
    };

    template <typename Key, typename Value>
    using TCachePtr = std::shared_ptr<TCache<Key, Value>>;

    template <typename Key, typename Value>
    class TContext: public std::enable_shared_from_this<TContext<Key, Value>> {
        using TBase = std::enable_shared_from_this<TContext<Key, Value>>;

    public:
        TContext(size_t totalSizeLimit, const TString& unistatName)
            : UnistatTotalSize_("cache_ctx." + unistatName + ".total_size", NUnistat::NSuffix::AVVV)
            , TotalSizeLimit_(totalSizeLimit)
            , TotalSize_(0)
        {
        }

        static TContextPtr<Key, Value> Create(size_t totalSizeLimit, const TString& unistatName) {
            return std::make_shared<NCache::TContext<Key, Value>>(totalSizeLimit, unistatName);
        }

        TCachePtr<Key, Value> CreateCache(size_t timeBucketCount,
                                          size_t keyBucketCount,
                                          const TString& unistatName) {
            Y_ENSURE(!Cleaner_, "Cleaning already started");

            auto res = std::make_shared<NCache::TCache<Key, Value>>(
                TBase::shared_from_this(),
                timeBucketCount,
                keyBucketCount,
                unistatName);
            Caches_.push_back(res);
            return res;
        }

        void StartCleaning(TDuration period) {
            Y_ENSURE(period != TDuration(), "cleaning period is null");
            Cleaner_ = std::make_unique<NUtils::TRegularTask>(
                [this]() { Clean(TInstant::Now()); },
                period,
                "cache_clean");
        }

        size_t GetTotalSize() const {
            return TotalSize_.load(std::memory_order_relaxed);
        }

        void AddUnistat(NUnistat::TBuilder& stats) {
            stats.AddRow(UnistatTotalSize_, GetTotalSize());
        }

    public: // for TCache
        bool CheckSize(size_t size) const {
            return GetTotalSize() + size < TotalSizeLimit_;
        }

        void IncrementSize(i64 size) {
            TotalSize_ += size;
        }

    public: // for tests
        void Clean(TInstant now) {
            size_t cleanedTotal = 0;

            do {
                cleanedTotal = 0;

                for (auto& w : Caches_) {
                    TCachePtr<Key, Value> cache = w.lock();

                    if (cache) {
                        const size_t cleaned = cache->TryClean(now);
                        cleanedTotal += cleaned;
                        TotalSize_ -= cleaned; // Allow to use cleaned memory as soon as posible
                    }
                }

            } while (cleanedTotal > 0);
        }

    private:
        const NUnistat::TName UnistatTotalSize_;
        const size_t TotalSizeLimit_;
        std::atomic<size_t> TotalSize_;

        std::vector<std::weak_ptr<TCache<Key, Value>>> Caches_;

        // must be last member
        std::unique_ptr<NUtils::TRegularTask> Cleaner_;
    };

    template <typename Key, typename Value>
    void TCache<Key, Value>::Put(Key&& key, Value&& value, TDuration lifeTime, TInstant now) {
        // There is the race here: size. It is ok for this case
        if (!Ctx_->CheckSize(GetSizeOf(key) + GetSizeOf(value))) {
            ++UnistatPutFailed_;
            return;
        }

        TTimeBasedBucket<Key, Value>& bucket = *Buckets_[GetWritingIdx().GetValue()];

        std::optional<i64> sizeDiff = bucket.Put(std::move(key), std::move(value), now + lifeTime);
        if (sizeDiff) {
            Ctx_->IncrementSize(*sizeDiff);
        } else {
            ++UnistatPutFailed_;
        }
    }
}
