#pragma once

#include "cache.h"

namespace NPassport::NCache {
    enum class EStatus {
        Hit,
        Miss,
        Unacceptable,
    };

    template <typename Key, typename Value>
    class TCacheHolder: TMoveOnly {
    public:
        TCacheHolder() = default;

        TCacheHolder(TCacheHolder&&) = default;
        TCacheHolder& operator=(TCacheHolder&&) = default;

        TCacheHolder(TCachePtr<Key, Value> cache, TDuration lifeTime, bool isDryRun)
            : Cache_(std::move(cache))
            , LifeTime_(lifeTime)
            , IsDryRun_(isDryRun)
        {
        }

        Value GetValue(const Key& key, EStatus& status) const {
            if (!IsEnabled()) {
                status = EStatus::Unacceptable;
                return {};
            }
            Y_ENSURE(key, "Key for cache cannot be empty");

            std::optional<Value> res = Cache_->Get(key);
            if (!res) {
                status = EStatus::Miss;
                return {};
            }

            status = EStatus::Hit;
            return IsDryRun_ ? Value() : *res;
        }

        void PutValue(Key&& key, Value&& data) {
            if (IsEnabled()) {
                Y_ENSURE(key, "Key for cache cannot be empty");
                Cache_->Put(std::move(key), std::move(data), LifeTime_);
            }
        }

        bool IsEnabled() const noexcept {
            return bool(Cache_);
        }

    private:
        TCachePtr<Key, Value> Cache_;
        TDuration LifeTime_;
        bool IsDryRun_;
    };
}
