#pragma once

#include "sizeof.h"

#include <util/datetime/base.h>

#include <memory>
#include <optional>
#include <shared_mutex>
#include <unordered_map>

namespace NPassport::NCache {
    template <typename Key, typename Val>
    class TKeyBasedBucket {
    public:
        struct TData {
            Val Value;
            TInstant ExpirationTime;
        };

        TKeyBasedBucket() {
            Data_.max_load_factor(0.5);
            Data_.reserve(4096);
        }

        std::optional<Val> Get(const Key& key, TInstant now) const {
            std::shared_lock lock(Mutex_);

            auto it = Data_.find(key);
            if (it == Data_.end()) {
                return {};
            }

            const TData& data = it->second;
            if (IsExpired(data.ExpirationTime, now)) {
                return {};
            }

            return data.Value;
        }

        i64 Put(Key&& key, Val&& value, TInstant expiration) {
            std::unique_lock lock(Mutex_);

            i64 res = 0;

            const size_t keySize = GetSizeOf(key);

            auto [it, inserted] = Data_.try_emplace(std::move(key), TData{});
            if (inserted) {
                res = keySize + GetSizeOf(value);
            } else {
                res = GetSizeOf(value) - GetSizeOf(it->second.Value);
            }

            it->second.Value = std::move(value);
            it->second.ExpirationTime = expiration;

            TotalSize_ += res;
            return res;
        }

        size_t Clean() {
            // thread protection is not required becase of unique_lock in TTimeBasedBucket
            Data_.clear();

            size_t res = TotalSize_;
            TotalSize_ = 0;
            return res;
        }

        static bool IsExpired(TInstant expiration, TInstant now) {
            return expiration <= now;
        }

        size_t GetTotalSize() const {
            return TotalSize_;
        }

    private:
        mutable std::shared_mutex Mutex_;
        std::unordered_map<Key, TData> Data_;
        size_t TotalSize_ = 0;
    };

    template <typename Key, typename Value>
    using TKeyBasedBucketPtr = std::unique_ptr<TKeyBasedBucket<Key, Value>>;
}
