#pragma once

#include "key_based_bucket.h"

#include <util/datetime/base.h>
#include <util/generic/yexception.h>

#include <atomic>
#include <optional>
#include <shared_mutex>
#include <vector>

namespace NPassport::NCache {
    template <typename Key, typename Value>
    class TTimeBasedBucket {
    public:
        TTimeBasedBucket(size_t keyBucketCount)
            : LatestExpiration_(TInstant())
        {
            Y_ENSURE(keyBucketCount > 0, "key count must be > 0");

            Data_.reserve(keyBucketCount);
            for (size_t idx = 0; idx < keyBucketCount; ++idx) {
                Data_.push_back(std::make_unique<TKeyBasedBucket<Key, Value>>());
            }
        }

        std::optional<Value> Get(const Key& key, TInstant now) const {
            if (IsEmpty(now)) {
                return {};
            }

            std::shared_lock lock(CleanerMutex_, std::defer_lock);
            if (!lock.try_lock()) { // impossible now
                return {};
            }

            return GetBucket(key).Get(key, now);
        }

        std::optional<i64> Put(Key&& key, Value&& value, TInstant expiration) {
            LatestExpiration_.store(expiration);

            std::shared_lock lock(CleanerMutex_, std::defer_lock);
            if (!lock.try_lock()) { // impossible now
                return {};
            }

            return GetBucket(key).Put(std::move(key), std::move(value), expiration);
        }

        std::optional<size_t> TryClean(TInstant now) {
            if (!IsEmpty(now)) {
                return {};
            }

            std::unique_lock lock(CleanerMutex_);

            size_t cleaned = 0;
            for (TKeyBasedBucketPtr<Key, Value>& buck : Data_) {
                cleaned += buck->Clean();
            }
            return cleaned;
        }

    protected:
        bool IsEmpty(TInstant now) const {
            return TKeyBasedBucket<Key, Value>::IsExpired(
                LatestExpiration_.load(std::memory_order_relaxed),
                now);
        }

        TKeyBasedBucket<Key, Value>& GetBucket(const Key& key) const {
            size_t idx = (std::hash<Key>{}(key) >> (sizeof(size_t) * 8 / 2)) % Data_.size();
            return *Data_.at(idx);
        }

    private:
        mutable std::shared_mutex CleanerMutex_;
        std::atomic<TInstant> LatestExpiration_;
        std::vector<TKeyBasedBucketPtr<Key, Value>> Data_;
    };

    template <typename Key, typename Value>
    using TTimeBasedBucketPtr = std::unique_ptr<TTimeBasedBucket<Key, Value>>;
}
