#include <passport/infra/libs/cpp/cache/cache.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/buffer.h>
#include <util/stream/buffer.h>

using namespace NPassport::NCache;

Y_UNIT_TEST_SUITE(Cache) {
    Y_UNIT_TEST(ContextIncrement) {
        TContext<int, int> ctx(100, "some_uni_name");

        UNIT_ASSERT_VALUES_EQUAL(0, ctx.GetTotalSize());
        UNIT_ASSERT(ctx.CheckSize(99));
        UNIT_ASSERT(!ctx.CheckSize(100));

        ctx.IncrementSize(10);
        UNIT_ASSERT_VALUES_EQUAL(10, ctx.GetTotalSize());
        UNIT_ASSERT(ctx.CheckSize(89));
        UNIT_ASSERT(!ctx.CheckSize(90));

        ctx.IncrementSize(85);
        UNIT_ASSERT_VALUES_EQUAL(95, ctx.GetTotalSize());
        UNIT_ASSERT(ctx.CheckSize(4));
        UNIT_ASSERT(!ctx.CheckSize(5));

        ctx.IncrementSize(10);
        UNIT_ASSERT_VALUES_EQUAL(105, ctx.GetTotalSize());
        UNIT_ASSERT(!ctx.CheckSize(0));
        UNIT_ASSERT(!ctx.CheckSize(1));
    }

    Y_UNIT_TEST(common) {
        auto ctx = std::make_shared<TContext<TString, TString>>(100, "some_uni_name");
        auto cache = ctx->CreateCache(3, 2048, "userinfo");

        TInstant now = TInstant::Now();

        UNIT_ASSERT(!cache->Get("kek", now));
        UNIT_ASSERT(!cache->Get("kek2", now));
        UNIT_ASSERT(!cache->Get("kek3", now));
        UNIT_ASSERT_VALUES_EQUAL(0, ctx->GetTotalSize());

        cache->Put("kek", "lol", TDuration::Hours(2), now);
        std::optional<TString> v = cache->Get("kek", now);
        UNIT_ASSERT(v);
        UNIT_ASSERT_VALUES_EQUAL("lol", *v);

        UNIT_ASSERT(!cache->Get("kek2", now));
        UNIT_ASSERT(!cache->Get("kek3", now));
        UNIT_ASSERT_VALUES_EQUAL(6, ctx->GetTotalSize());

        cache->Put("kek", "lol12345", TDuration::Hours(2), now);
        UNIT_ASSERT_VALUES_EQUAL(11, ctx->GetTotalSize());
        v = cache->Get("kek", now);
        UNIT_ASSERT(v);
        UNIT_ASSERT_VALUES_EQUAL("lol12345", *v);
        UNIT_ASSERT(!cache->Get("kek2", now));

        cache->Put("kek2", "0123456789", TDuration::Hours(2), now);
        UNIT_ASSERT_VALUES_EQUAL(25, ctx->GetTotalSize());

        v = cache->Get("kek", now);
        UNIT_ASSERT(v);
        UNIT_ASSERT_VALUES_EQUAL("lol12345", *v);
        v = cache->Get("kek2", now);
        UNIT_ASSERT(v);
        UNIT_ASSERT_VALUES_EQUAL("0123456789", *v);
        UNIT_ASSERT(!cache->Get("kek3", now));

        cache->Put("kek3", TString(71, 'a'), TDuration::Hours(2), now);
        UNIT_ASSERT_VALUES_EQUAL(25, ctx->GetTotalSize());
        UNIT_ASSERT(!cache->Get("kek3", now));

        cache->Put("kek2", "0123", TDuration::Hours(2), now);
        v = cache->Get("kek", now);
        UNIT_ASSERT(v);
        UNIT_ASSERT_VALUES_EQUAL("lol12345", *v);
        v = cache->Get("kek2", now);
        UNIT_ASSERT(v);
        UNIT_ASSERT_VALUES_EQUAL("0123", *v);
        UNIT_ASSERT(!cache->Get("kek3", now));
        UNIT_ASSERT_VALUES_EQUAL(19, ctx->GetTotalSize());

        // after 2 hours
        UNIT_ASSERT(!cache->Get("kek", now + TDuration::Hours(2)));
        UNIT_ASSERT(!cache->Get("kek2", now + TDuration::Hours(2)));
        UNIT_ASSERT(!cache->Get("kek3", now + TDuration::Hours(2)));
        UNIT_ASSERT_VALUES_EQUAL(19, ctx->GetTotalSize());

        ctx->Clean(now + TDuration::Hours(2));
        UNIT_ASSERT(cache->Get("kek", now));
        UNIT_ASSERT(cache->Get("kek2", now));
        UNIT_ASSERT(!cache->Get("kek3", now));
        UNIT_ASSERT(!cache->Get("kek", now + TDuration::Hours(2)));
        UNIT_ASSERT(!cache->Get("kek2", now + TDuration::Hours(2)));
        UNIT_ASSERT(!cache->Get("kek3", now + TDuration::Hours(2)));
        UNIT_ASSERT_VALUES_EQUAL(19, ctx->GetTotalSize());

        ctx->Clean(now + TDuration::Hours(2));
        UNIT_ASSERT(cache->Get("kek"));
        UNIT_ASSERT(cache->Get("kek2"));
        UNIT_ASSERT(!cache->Get("kek3", now));
        UNIT_ASSERT(!cache->Get("kek", now + TDuration::Hours(2)));
        UNIT_ASSERT(!cache->Get("kek2", now + TDuration::Hours(2)));
        UNIT_ASSERT(!cache->Get("kek3", now + TDuration::Hours(2)));
        UNIT_ASSERT_VALUES_EQUAL(19, ctx->GetTotalSize());

        ctx->Clean(now + TDuration::Hours(2));
        UNIT_ASSERT(!cache->Get("kek"));
        UNIT_ASSERT(!cache->Get("kek2"));
        UNIT_ASSERT(!cache->Get("kek3", now));
        UNIT_ASSERT(!cache->Get("kek", now + TDuration::Hours(2)));
        UNIT_ASSERT(!cache->Get("kek2", now + TDuration::Hours(2)));
        UNIT_ASSERT(!cache->Get("kek3", now + TDuration::Hours(2)));
        UNIT_ASSERT_VALUES_EQUAL(0, ctx->GetTotalSize());
    }

    Y_UNIT_TEST(tryClean) {
        auto ctx = std::make_shared<TContext<TString, TString>>(100, "some_uni_name");
        auto cache = ctx->CreateCache(3, 2048, "userinfo");

        TInstant now = TInstant::Now();
        UNIT_ASSERT_VALUES_EQUAL(0, cache->GetWritingIdx().GetValue());

        cache->Put("kek", "lol12345", TDuration::Hours(2), now);
        cache->Put("kek2", "0123456789", TDuration::Hours(2), now);
        UNIT_ASSERT_VALUES_EQUAL(25, ctx->GetTotalSize());

        UNIT_ASSERT_VALUES_EQUAL(0, cache->GetWritingIdx().GetValue());

        UNIT_ASSERT_VALUES_EQUAL(0, cache->TryClean(now));
        UNIT_ASSERT_VALUES_EQUAL(1, cache->GetWritingIdx().GetValue());

        UNIT_ASSERT_VALUES_EQUAL(0, cache->TryClean(now));
        UNIT_ASSERT_VALUES_EQUAL(2, cache->GetWritingIdx().GetValue());

        UNIT_ASSERT_VALUES_EQUAL(0, cache->TryClean(now)); // idx==0 is not expired yet
        UNIT_ASSERT_VALUES_EQUAL(2, cache->GetWritingIdx().GetValue());

        UNIT_ASSERT_VALUES_EQUAL(25, cache->TryClean(now + TDuration::Hours(2)));
        UNIT_ASSERT_VALUES_EQUAL(0, cache->GetWritingIdx().GetValue());

        UNIT_ASSERT_VALUES_EQUAL(25, ctx->GetTotalSize());
    }
}
