#pragma once

#include "db_info.h"
#include "db_pool.h"
#include "exception.h"
#include "query.h"
#include "result.h"
#include "misc/handle.h"

#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/datetime/base.h>
#include <util/generic/noncopyable.h>

namespace NPassport::NDbPool {
    class TBlockingHandle: public TNonCopyable {
    public:
        TBlockingHandle(TBlockingHandle&&) = delete;
        TBlockingHandle& operator=(TBlockingHandle&&) = delete;

        explicit TBlockingHandle(TDbPool& pool)
            : Pool_(pool)
            , Sqlh_(Pool_.Get())
        {
            if (!Sqlh_) {
                throw TCantGetConnection(Pool_.GetDbInfo()) << Message("Can't get connection");
            }
        }

        ~TBlockingHandle() {
            Pool_.Put(std::move(Sqlh_));
        }

        std::unique_ptr<TResult> Query(TQuery&& query, std::optional<TDuration> customTimeout = {}) {
            return Sqlh_->Query(std::move(query), customTimeout);
        }

        const TDbInfo& GetDbInfo() const {
            return Sqlh_->GetDbInfo();
        }

        TString EscapeQueryParam(const TStringBuf param) {
            if (!Sqlh_) {
                throw TException(Pool_.GetDbInfo()) << Message("missing handle");
            }

            return Sqlh_->EscapeQueryParam(param);
        }

    private:
        TString Message(const TStringBuf msg) const {
            return NPassport::NUtils::CreateStr(msg, " (DSN: ", Pool_.GetDbInfo().Serialized, ')');
        }

    private:
        TDbPool& Pool_;
        std::unique_ptr<THandle> Sqlh_;
    };

    class TNonBlockingHandle: public TMoveOnly {
    public:
        TNonBlockingHandle()
            : Start_(TInstant::Now())
        {
        }

        explicit TNonBlockingHandle(TDbPool& pool)
            : Pool_(&pool)
            , Sqlh_(Pool_->Get())
            , Start_(TInstant::Now())
        {
            CheckHandle();
        }

        TNonBlockingHandle(TNonBlockingHandle&& o) noexcept
            : Pool_(o.Pool_)
            , Sqlh_(std::move(o.Sqlh_))
            , Start_(o.Start_)
        {
        }

        TNonBlockingHandle& operator=(TNonBlockingHandle&& o) noexcept {
            if (this == &o) {
                return *this;
            }
            std::swap(Pool_, o.Pool_);
            std::swap(Sqlh_, o.Sqlh_);
            std::swap(Start_, o.Start_);
            return *this;
        }

        ~TNonBlockingHandle() {
            if (Sqlh_) {
                // 'bad' handles already finished work - don't try wait them
                if (!Sqlh_->Bad()) {
                    // TODO put handle which in progress to 'long query' queue, do not wait it
                    Sqlh_->WaitToFinish();
                }

                Pool_->Put(std::move(Sqlh_));
            }
        }

        void SendQuery(TQuery&& query, std::optional<TDuration> customTimeout = {}) {
            CheckHandle();
            Sqlh_->NonBlockingQuery(std::move(query), customTimeout);
        }

        std::unique_ptr<TResult> WaitResult() {
            CheckHandle();
            return Sqlh_->WaitResult();
        }

        operator bool() const {
            return bool(Sqlh_);
        }

        TInstant StartTime() const {
            return Start_;
        }

        const TDbInfo& GetDbInfo() const {
            CheckHandle();
            return Sqlh_->GetDbInfo();
        }

        TString EscapeQueryParam(const TStringBuf param) {
            if (!Sqlh_) {
                throw TException(Pool_->GetDbInfo()) << Message("missing handle");
            }

            return Sqlh_->EscapeQueryParam(param);
        }

    private:
        TString Message(const TStringBuf msg) const {
            return NPassport::NUtils::CreateStr(msg, " (DSN: ", Pool_->GetDbInfo().Serialized, ')');
        }

        void CheckHandle() const {
            if (!Sqlh_) {
                throw TCantGetConnection(Pool_->GetDbInfo()) << Message("Can't get connection");
            }
        }

    private:
        TDbPool* Pool_ = nullptr;
        std::unique_ptr<THandle> Sqlh_;
        TInstant Start_;
    };
}
