#pragma once

#include "counter.h"

#include <passport/infra/libs/cpp/dbpool/db_info.h>
#include <passport/infra/libs/cpp/dbpool/db_pool_ctx.h>
#include <passport/infra/libs/cpp/dbpool/destination.h>

#include <passport/infra/libs/cpp/utils/log/global.h>

#include <library/cpp/threading/future/future.h>

#include <util/datetime/base.h>
#include <util/generic/string.h>

#include <memory>
#include <thread>

namespace NPassport::NDbPool {
    class TQuery;
    class TResult;
    class TWorkerShared;

    struct THandleSettings {
        TDestinationPtr Dsn;
        TDbHost DbHost;
        TDuration ConnectionTimeout;
        TDuration QueryTimeout;
        bool FetchStatusOnPing = false;

        TQueryOptsPtr DefaultQueryOpts = std::make_shared<TQueryOpts>();
    };

    struct THandleUnistatCtx {
        TDbPoolCtx::TTimeStat TimeStats;
        TCountersPtr Counters;

        TDbPoolCtx::TTimeStat PoolTimeStats;
        TCountersPtr PoolCounters;
    };

    using THandleInitError = std::optional<TString>;

    struct TResponse {
        bool IsOk = false;
        TString Randid;
        TString Error;
        std::unique_ptr<TResult> Result;

        TInstant Start = TInstant::Now();
    };

    class THandle {
    public:
        struct TNonblockingInit {
            std::unique_ptr<THandle> Handle;
            TString Randid;
            NThreading::TFuture<THandleInitError> InitionError;
            TInstant Start = TInstant::Now();
        };

        static TNonblockingInit CreateNonblocking(
            const THandleSettings& settings,
            THandleUnistatCtx unistat,
            TDbPoolLog logger,
            size_t hostIdx = -1);

        ~THandle();

        const TDbInfo& GetDbInfo() const;
        size_t GetHostIdx() const;

        bool Ping();

        using TSubscribeFunc = std::function<void(NThreading::TFuture<TResponse>)>;
        void StartNonblockingPing(TSubscribeFunc func = {});

        TString EscapeQueryParam(const TStringBuf s);

        std::unique_ptr<TResult> Query(TQuery&& q, std::optional<TDuration> customTimeout = {});

        void NonBlockingQuery(TQuery&& query, std::optional<TDuration> customTimeout = {});
        std::unique_ptr<TResult> WaitResult(std::optional<TDuration> customTimeout = {});

        std::unique_ptr<TResult> CheckPingFinished();

        void WaitToFinish() noexcept;
        bool TryMakeClear();
        bool Bad() const;

    private:
        bool WaitToFinishImpl() noexcept;
        TString ErrMsgTimeout(TDuration timeout) const;

    private:
        class TProcHolder {
        public:
            TProcHolder(std::shared_ptr<TWorkerShared> shared);
            ~TProcHolder();

        private:
            std::shared_ptr<TWorkerShared> Shared_; // shared_ptr is redundancy for refactoring-safety
            std::thread Thread_;
        };
        friend class TProcHolder;

    private:
        // Nonblocking ctor
        THandle(const THandleSettings& settings,
                THandleUnistatCtx unistat,
                TDbPoolLog logger);

        NThreading::TFuture<TResponse> NonBlockingQueryImpl(std::optional<TDuration> customTimeout = {});

        static void Proc(std::shared_ptr<TWorkerShared> shared);

    private:
        const TDuration DefaultQueryTimeout_;
        const TDbPoolLog Log_;
        const TDbInfo DbInfo_;
        std::shared_ptr<TWorkerShared> Shared_;
        bool Badbit_ = false;
        size_t HostIdx_ = 0UL;

        const TQueryOptsPtr DefaultQueryOpts_;

        // must be last
        const TProcHolder ProcHolder_;
    };
}
