#include "host.h"

#include "pinger.h"

#include <passport/infra/libs/cpp/dbpool/exception.h>

#include <passport/infra/libs/cpp/json/writer.h>
#include <passport/infra/libs/cpp/unistat/time_stat.h>
#include <passport/infra/libs/cpp/utils/string/split.h>

namespace NPassport::NDbPool {
    THost::THost(const THostSettings& settings,
                 THostUnistatCtx unistat,
                 TDbPoolLog log,
                 size_t idx)
        : Settings_(settings)
        , Log_(log)
        , DbInfo_(*Settings_.Dsn, Settings_.DbHost)
        , State_(Log_, Settings_.FailThreshold, DbInfo_.Serialized, TPoolState::ELogLevel::Warn)
        , Unistat_(THandleUnistatCtx{
              .PoolTimeStats = unistat.PoolTimeStats,
              .PoolCounters = unistat.PoolCounters,
          })
        , Idx_(idx)
    {
        Log_.Debug() << DbInfo_
                     << " Host will start with " << Settings_.InitialSize << " handles. id="
                     << DbInfo_.DisplayName;

        // drop ipv6 brackets for HTTP
        TString hostInSignal = NUtils::ReplaceAny(Settings_.DbHost.Host, "[]", "");
        // YASM doesn't support ':' in signal name
        hostInSignal = NUtils::ReplaceAny(hostInSignal, ":", "_");

        const TString signalId = NUtils::CreateStr(
            Settings_.Dsn->DisplayName, // to identify pool
            "/",                        // delimeter which impossible in db name and in host
            hostInSignal,               // ip address or hostname
            "/"                         // delimeter to determine 'host' end
        );

        Unistat_.TimeStats = THost::MakeStats(*Settings_.Dsn, signalId);
        Unistat_.Counters = std::make_shared<TCounters>(signalId, Settings_.InitialSize);
    }

    THost::~THost() {
        Pool_.Clear();
        BadPool_.Clear();

        // Join threads to avoid segfault on stop:
        //   this thread can touch global state which could be destroyed already
        while (GetHandlesInfo().Total > 0) {
            Sleep(TDuration::MilliSeconds(100));
        }
    }

    const TDbHost& THost::GetDbHost() const {
        return Settings_.DbHost;
    }

    const TDbInfo& THost::GetDbInfo() const {
        return DbInfo_;
    }

    bool THost::IsOk() const {
        return TPoolState::IsOk(State_, DbInfo_, nullptr);
    }

    THandlesInfo THost::GetHandlesInfo() const {
        return Unistat_.Counters->GetHandlesInfo();
    }

    void THost::AddUnistatExtended(NUnistat::TBuilder& builder) const {
        Unistat_.Counters->AddUnistat(builder);
        Unistat_.TimeStats->AddUnistat(builder);
    }

    void THost::TryPing(TDuration getTimeout) {
        // Attempt to avoid parallel initialization of several instances of DBPool
        WaitForHandles();

        // Now pool should be not empty. So we can ping backend
        PingOneHandle(getTimeout);
    }

    THandlesInfo THost::GetExtendedStats(NJson::TObject& out) const {
        NJson::TObject host(out, Settings_.DbHost.Host);

        TString err;
        const bool ok = TPoolState::IsOk(State_, DbInfo_, &err);
        host.Add("status", ok ? "OK" : "ERROR");
        host.Add("error", ok ? TString() : err);

        THandlesInfo counters = GetHandlesInfo();
        host.Add("config_size", counters.ConfigCount);
        host.Add("workers_idle", counters.Idle);
        host.Add("workers_in_use", counters.InUse);
        host.Add("workers_dying", counters.Dying);

        return counters;
    }

    std::unique_ptr<THandle> THost::TryGetNonblocking() {
        if (Pool_.ThreadSafeSize() == 0) { // optimization
            ++Unistat_.Counters->CantGetConnection;
            State_.TryMakeIfNotMatch(TPoolState::Down, TPoolState::ForceDown);
            return {};
        }

        std::unique_lock lock(Mutex_);

        if (Pool_.IsEmpty()) {
            ++Unistat_.Counters->CantGetConnection;
            State_.TryMakeIfNotMatch(TPoolState::Down, TPoolState::ForceDown);
            return {};
        }

        --Unistat_.Counters->IdleHandles;
        --Unistat_.PoolCounters->IdleHandles;

        return Pool_.Pop();
    }

    EPutHandleStatus THost::Put(std::unique_ptr<THandle> handle) {
        Y_VERIFY(handle, "put(NULL): %s", DbInfo_.Serialized.c_str());

        if (handle->Bad()) {
            Log_.Debug() << DbInfo_ << " put(bad) to bad pool";
            --Unistat_.Counters->GoodHandles;
            --Unistat_.PoolCounters->GoodHandles;

            std::unique_lock lock(MutexBadPool_);
            BadPool_.Push(std::move(handle));

            return EPutHandleStatus::Bad;
        }

        ++Unistat_.Counters->IdleHandles;
        ++Unistat_.PoolCounters->IdleHandles;

        std::unique_lock lock(Mutex_);
        if (Unistat_.Counters->GoodHandles.GetValue() >= Unistat_.Counters->ConfigSize.GetValue()) {
            State_.TryMakeIfNotMatch(TPoolState::Up, TPoolState::ForceDown);
        }
        Pool_.Push(std::move(handle));

        return EPutHandleStatus::Ok;
    }

    THandle::TNonblockingInit THost::CreateHandle() const {
        return THandle::CreateNonblocking(
            THandleSettings{
                .Dsn = Settings_.Dsn,
                .DbHost = Settings_.DbHost,
                .ConnectionTimeout = Settings_.ConnectionTimeout,
                .QueryTimeout = Settings_.QueryTimeout,
                .FetchStatusOnPing = Settings_.FetchStatusOnPing,
                .DefaultQueryOpts = Settings_.DefaultQueryOpts,
            },
            Unistat_,
            Log_,
            Idx_);
    }

    void THost::PutNewHandle(std::unique_ptr<THandle> handle) {
        ++Unistat_.Counters->IdleHandles;
        ++Unistat_.PoolCounters->IdleHandles;
        ui64 good = ++Unistat_.Counters->GoodHandles;
        ++Unistat_.PoolCounters->GoodHandles;
        ui64 leftCount = TCounters::Diff(Unistat_.Counters->ConfigSize.GetValue(), good);

        std::unique_lock lock(Mutex_);
        Pool_.Push(std::move(handle));

        if (leftCount == 0) {
            State_.TryMakeIfNotMatch(TPoolState::Up, TPoolState::ForceDown);
        }
    }

    void THost::CleanupOneBad() {
        if (BadPool_.ThreadSafeSize() == 0) { // optimization
            return;
        }

        std::unique_ptr<THandle> h;
        {
            std::unique_lock lock(MutexBadPool_);
            if (BadPool_.IsEmpty()) {
                return;
            }

            h = BadPool_.Pop();
        }

        if (h->TryMakeClear()) {
            Log_.Debug() << DbInfo_ << " Back handle to pool from bad pool. id=" << DbInfo_.DisplayName;
            PutNewHandle(std::move(h));
        } else {
            Log_.Debug() << DbInfo_ << " Delete handle from bad pool. id=" << DbInfo_.DisplayName;
        }
    }

    void THost::SetHandlesCount(ui64 size) {
        ui64 prevSize = Unistat_.Counters->ConfigSize.exchange(size, std::memory_order_relaxed);

        if (prevSize != size) {
            Log_.Debug() << DbInfo_
                         << " Host changed handle count: " << prevSize << "->" << size
                         << ". id=" << DbInfo_.DisplayName;
        }
    }

    void THost::ForceDown(bool value) {
        if (value) {
            State_.TryMake(TPoolState::ForceDown);
        } else {
            State_.TryMakeIfMatch(
                Unistat_.Counters->GoodHandles.GetValue() >= Unistat_.Counters->ConfigSize.GetValue()
                    ? TPoolState::Up
                    : TPoolState::Down,
                TPoolState::ForceDown);
        }
    }

    void THost::OnPingException() {
        if (Settings_.FetchStatusOnPing) {
            // ForceDown: only explicit OK from backend should open it for traffic
            State_.TryMake(TPoolState::ForceDown);
        }
    }

    TDbPoolCtx::TTimeStat THost::MakeStats(const TDestination& dsn, const TString& id) {
        NUnistat::TTimeStat::TBounds bounds;

        auto it = dsn.Extended.find("time_stats_bounds");
        if (it == dsn.Extended.end()) {
            bounds = NUnistat::TTimeStat::CreateDefaultBounds();
        } else {
            NUtils::Transform(it->second, ',', [&bounds](const TStringBuf buf) -> void {
                unsigned tmp;
                Y_ENSURE(TryIntFromString<10>(buf, tmp), "Bounds must be unsinged integers");
                bounds.push_back(TDuration::MilliSeconds(tmp));
            });
        }

        return std::make_shared<NUnistat::TTimeStat>(
            "dbpool." + id + ".responsetime",
            std::move(bounds));
    }

    void THost::WaitForHandles() {
        const TInstant deadline = TInstant::Now() + Settings_.TimeToInit;

        while (Unistat_.Counters->GoodHandles.GetValue() < Unistat_.Counters->ConfigSize.GetValue() &&
               TInstant::Now() < deadline) {
            Sleep(TDuration::MilliSeconds(100));
        }
    }

    void THost::PingOneHandle(TDuration getTimeout) {
        TPingGuard g(*this, TryGetNonblocking());

        const TInstant deadline = TInstant::Now() + getTimeout;
        while (!g.Handle && TInstant::Now() < deadline) {
            Sleep(TDuration::MilliSeconds(100));
            g.Handle = TryGetNonblocking();
        }

        if (!g.Handle) {
            throw TCantGetConnection(DbInfo_) << "Can't get connection";
        }

        if (!g.Handle->Ping()) {
            throw TException(DbInfo_) << "Could not initialize connection: host was explicitly disabled";
        }
    }
}
