#include "http-driver.h"

#include <passport/infra/libs/cpp/dbpool/result.h>

#include <passport/infra/libs/cpp/utils/ipaddr.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/format.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/generic/string.h>

#include <algorithm>
#include <functional>

namespace NPassport::NDbPool {
    static const TString CONTENT_TYPE = "content-type";

    static const char* const KEEP_ALIVE = "Connection: Keep-Alive";
    static const TString EMPTY_;

    TString THttpResult::ToString() const {
        TStringStream res;
        res << "Http status: " << Status << ", content-type: " << ContentType << Endl;
        res << "Http headers:" << Endl;
        for (const auto& [name, value] : Headers.Data) {
            res << "    " << name << " : " << value << Endl;
        }

        res << "Http body:" << Endl;
        res << Body.Data;

        return res.Str();
    }

    THttpDriver::THttpDriver()
        : Curl_(nullptr, curl_easy_cleanup)
    {
    }

    bool THttpDriver::Connect(const TString& host,
                              int port,
                              const TString&,
                              const TZtStringBuf,
                              const TString&,
                              TDuration connectTimeout,
                              TDuration,
                              const TExtendedArgs& args,
                              bool) {
        BaseUrl_ = BuildBaseUrl(host, port);
        Curl_.reset(curl_easy_init());

        if (!Curl_) {
            return false;
        }

        // set up global options
        CheckedSetopt(CURLOPT_NOSIGNAL, (long)1);
        CheckedSetopt(CURLOPT_NOPROGRESS, (long)1);
        CheckedSetopt(CURLOPT_FORBID_REUSE, (long)0);
        CheckedSetopt(CURLOPT_FRESH_CONNECT, (long)0);

        CheckedSetopt(CURLOPT_WRITEFUNCTION, &WriteData);
        CheckedSetopt(CURLOPT_HEADERFUNCTION, &WriteHeader);

        CheckedSetopt(CURLOPT_CONNECTTIMEOUT_MS, (long)connectTimeout.MilliSeconds());

        if (BaseUrl_.StartsWith("https")) {
            CheckedSetopt(CURLOPT_SSL_VERIFYPEER, true);
            CheckedSetopt(CURLOPT_SSL_VERIFYHOST, 2L);
            CheckedSetopt(CURLOPT_CAPATH, "/etc/ssl/certs");
        }

        auto it = args.find("ping_path");
        PingPath_ = (it == args.end()) ? "/ping" : it->second;

        it = args.find("header_host");
        if (it != args.end()) {
            HeaderHost_ = "Host: " + it->second;
        }

        AddDefaultHeaders(DefaultHeaders_);

        // Ping once to perform actual connect.
        // Connect timeout here allows to perform connect despite of any lags:
        //   TLS-handshake or any other steps
        return (bool)Ping(connectTimeout);
    }

    std::unique_ptr<TResult> THttpDriver::Query(const TQuery& q, TDuration queryTimeout) {
        RetCode_ = CURLE_FAILED_INIT;
        if (!Perform(q.Query(), q.HttpHeaders(), q.HttpBody(), q.HttpMethod(), queryTimeout)) {
            return {};
        }
        return StoreResult();
    }

    std::unique_ptr<TResult> THttpDriver::StoreResult() {
        TRow statusWithContentType;
        statusWithContentType.reserve(2);
        statusWithContentType.emplace_back(IntToString<10>(Result_->Status));
        statusWithContentType.emplace_back(std::move(Result_->ContentType));

        TRow headers;
        headers.reserve(Result_->Headers.Data.size());
        for (auto& p : Result_->Headers.Data) {
            headers.emplace_back(TString(p.first));
            headers.emplace_back(std::move(p.second));
        }

        TRow body;
        body.emplace_back(std::move(Result_->Body.Data));

        TTable table;
        table.reserve(3);
        table.push_back(std::move(statusWithContentType));
        table.push_back(std::move(headers));
        table.push_back(std::move(body));

        return std::make_unique<TResult>(std::move(table));
    }

    int THttpDriver::ErrNum() const {
        return RetCode_;
    }

    TString THttpDriver::Error() {
        if (RetCode_ == CURLE_OK && Result_) {
            return "HTTP-code=" + IntToString<10>(Result_->Status);
        }

        const char* pstr = curl_easy_strerror(RetCode_);
        return pstr ? TString(pstr) : TString();
    }

    TString THttpDriver::EscapeQueryParam(const TStringBuf s) const {
        TString res(s);
        std::replace(res.begin(), res.vend(), ' ', '_');
        return res;
    }

    std::unique_ptr<TResult> THttpDriver::Ping(TDuration queryTimeout) {
        if (!(Perform(PingPath_, {}, EMPTY_, EMPTY_, queryTimeout) && Result_->Status == 200)) {
            return {};
        }

        // TODO: fetch weight from backend
        TTable res;
        TRow& row = res.emplace_back();
        row.emplace_back("1");

        return std::make_unique<TResult>(std::move(res), 0);
    }

    TString THttpDriver::BuildBaseUrl(const TString& host, int port) {
        TString res;

        if (host.StartsWith("http")) {
            res = host;
        } else {
            NUtils::TIpAddr ip;
            if (ip.Parse(host) && ip.IsIpv6()) {
                res = NUtils::CreateStr("http://[", host, "]");
            } else {
                res = NUtils::CreateStr("http://", host);
            }
        }

        if (port) {
            NUtils::Append(res, ':', port);
        }

        return res;
    }

    template <typename T>
    void THttpDriver::CheckedSetopt(CURLoption opt, T val) {
        CURLcode retCode = curl_easy_setopt(Curl_.get(), opt, val);
        if (retCode != CURLE_OK) {
            RetCode_ = retCode;
            throw yexception() << curl_easy_strerror(retCode);
        }
    }

    size_t THttpDriver::WriteData(char* ptr, size_t size, size_t nmemb, void* userdata) {
        THttpResult::TResponseBody* body = (THttpResult::TResponseBody*)userdata;
        size_t bytes = size * nmemb;

        if (bytes) {
            body->Data.append(ptr, bytes);
        }

        return bytes;
    }

    size_t THttpDriver::WriteHeader(char* ptr, size_t size, size_t nmemb, void* userdata) {
        THttpResult::TResponseHeaders* head = (THttpResult::TResponseHeaders*)userdata;
        size_t bytes = size * nmemb;

        if (bytes == 0) {
            return bytes;
        }

        TStringBuf header(ptr, bytes);
        size_t pos = header.find(':');

        if (pos == TStringBuf::npos) {
            return bytes;
        }

        TStringBuf name = header.substr(0, pos);
        TStringBuf value = header.substr(pos + 1);
        NUtils::Trim(name);
        NUtils::Trim(value);

        std::pair<TString, TString> item = {TString(name), TString(value)};
        NUtils::Tolower(item.first);

        head->Data.insert(std::move(item));

        return bytes;
    }

    bool THttpDriver::Perform(const TString& path,
                              const TQuery::THttpHeaders& headers,
                              const TString& body,
                              const TString& method,
                              TDuration queryTimeout) {
        std::shared_ptr<THttpResult> res = std::make_shared<THttpResult>();

        const TString url = NUtils::CreateStr(
            BaseUrl_,
            path.StartsWith('/') ? "" : "/",
            path);

        // set up request-specific params
        CheckedSetopt(CURLOPT_URL, url.c_str());
        CheckedSetopt(CURLOPT_WRITEDATA, &(res->Body));
        CheckedSetopt(CURLOPT_WRITEHEADER, &(res->Headers));
        CheckedSetopt(CURLOPT_CUSTOMREQUEST, method.empty() ? nullptr : method.c_str());
        // `CURLOPT_TIMEOUT_MS` must be specified to avoid infinite waiting
        CheckedSetopt(CURLOPT_TIMEOUT_MS, (long)queryTimeout.MilliSeconds());

        if (body.empty()) {
            CheckedSetopt(CURLOPT_POST, false);
        } else {
            CheckedSetopt(CURLOPT_POST, true);
            CheckedSetopt(CURLOPT_POSTFIELDS, body.c_str());
            CheckedSetopt(CURLOPT_POSTFIELDSIZE, body.size());
        }

        //    process Headers
        TRequestHeaders heads;
        if (headers.empty()) {
            CheckedSetopt(CURLOPT_HTTPHEADER, DefaultHeaders_.Get());
        } else {
            AddDefaultHeaders(heads);
            for (const TString& h : headers) {
                heads.Append(h.c_str());
            }
            CheckedSetopt(CURLOPT_HTTPHEADER, heads.Get());
        }

        RetCode_ = curl_easy_perform(Curl_.get());
        if (RetCode_ != CURLE_OK) {
            return false;
        }

        RetCode_ = curl_easy_getinfo(Curl_.get(), CURLINFO_RESPONSE_CODE, &(res->Status));
        if (RetCode_ != CURLE_OK) {
            return false;
        }

        // fill content_type
        THttpResult::THeaderMap::iterator i = res->Headers.Data.find(CONTENT_TYPE);
        if (i != res->Headers.Data.end()) {
            // cut until ; or string end end
            res->ContentType = i->second.substr(0, i->second.find(';'));
            NPassport::NUtils::Trim(res->ContentType);
        }

        Result_ = std::move(res);

        return RetCode_ == CURLE_OK;
    }

    void THttpDriver::AddDefaultHeaders(TRequestHeaders& headers) {
        headers.Append(KEEP_ALIVE);

        if (HeaderHost_) {
            headers.Append(HeaderHost_.c_str());
        }
    }

    // static helper to do global init/cleanup
    class TCurlInitializer {
    public:
        TCurlInitializer() {
            curl_global_init(CURL_GLOBAL_ALL);
        }

        ~TCurlInitializer() {
            curl_global_cleanup();
        }
    };

    static const TCurlInitializer CURL_INITIALIZER;
}
