#pragma once

#include "driver.h"

#include <passport/infra/libs/cpp/dbpool/query.h>

#include <contrib/libs/curl/include/curl/curl.h>

#include <util/generic/noncopyable.h>
#include <util/generic/string.h>

#include <map>
#include <memory>

namespace NPassport::NDbPool {
    class THttpResult {
    public:
        using THeaderMap = std::multimap<TString, TString>;

        class TResponseHeaders {
        public:
            THeaderMap Data;
        };

        class TResponseBody {
        public:
            TString Data;
        };

        TString ToString() const;

        long Status = 0L;
        TString ContentType;
        TResponseHeaders Headers;
        TResponseBody Body;
    };

    class TRequestHeaders: private TNonCopyable {
    public:
        TRequestHeaders() = default;

        ~TRequestHeaders() {
            Clear();
        }

        const curl_slist* Get() const {
            return Headers_;
        }

        void Append(const char* header) {
            Headers_ = curl_slist_append(Headers_, header);
        }

        void Clear() {
            if (Headers_ != nullptr) {
                curl_slist_free_all(Headers_);
                Headers_ = nullptr;
            }
        }

    private:
        curl_slist* Headers_ = nullptr;
    };

    class THttpDriver: public IDriver {
    public:
        THttpDriver();

        bool Connect(const TString& host,
                     int port,
                     const TString& user,
                     const TZtStringBuf pwd,
                     const TString& db,
                     TDuration connectTimeout,
                     TDuration queryTimeout,
                     const TExtendedArgs& args,
                     bool fetchStatusOnPing) override;

        // HTTP response stored as a table:
        // 1 row: http status (string), content-type (string)
        // 2nd row: all headers in a vector in pairs
        // 3rd row: first element - response body
        std::unique_ptr<TResult> Query(const TQuery& q, TDuration queryTimeout) override;

        std::unique_ptr<TResult> StoreResult();

        int ErrNum() const override;
        TString Error() override;
        TString EscapeQueryParam(const TStringBuf) const override;
        std::unique_ptr<TResult> Ping(TDuration queryTimeout) override;

        static TString BuildBaseUrl(const TString& host, int port);

    private:
        using TCurlHandle = std::unique_ptr<CURL, std::function<void(CURL*)>>;

        // curl_easy_setopt with result check and exception
        template <typename T>
        void CheckedSetopt(CURLoption opt, T val);

        // curl callbacks
        static size_t WriteData(char* ptr, size_t size, size_t nmemb, void* userdata);
        static size_t WriteHeader(char* ptr, size_t size, size_t nmemb, void* userdata);

        // final setup and actual communication
        bool Perform(const TString& path,
                     const TQuery::THttpHeaders& headers,
                     const TString& body,
                     const TString& method,
                     TDuration queryTimeout);

        void AddDefaultHeaders(TRequestHeaders& headers);

        TCurlHandle Curl_;
        TString BaseUrl_;
        TRequestHeaders DefaultHeaders_;

        CURLcode RetCode_ = CURLE_OK;
        std::shared_ptr<THttpResult> Result_;
        TString PingPath_;
        TString HeaderHost_;
    };
}
