#include "poolstate.h"

#include <passport/infra/libs/cpp/dbpool/db_info.h>

#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/stream/format.h>

namespace NPassport::NDbPool {
    TPoolState::TPoolState(TDbPoolLog log,
                           TDuration failThreshold,
                           const TString& dsn,
                           ELogLevel logLevel)
        : Log_(log)
        , FailThreshold_(failThreshold)
        , Dsn_(dsn)
        , LogLevel_(logLevel)
    {
    }

    void TPoolState::TryMake(EState state, TInstant now) {
        TryMakeImpl(state, state, true, now);
    }

    void TPoolState::TryMakeIfMatch(EState toSet, EState expected, TInstant now) {
        TryMakeImpl(toSet, expected, false, now);
    }

    void TPoolState::TryMakeIfNotMatch(EState toSet, EState expected, TInstant now) {
        TryMakeImpl(toSet, expected, true, now);
    }

    void TPoolState::TryMakeImpl(EState toSet, EState expected, bool skipIfExpectedMathched, TInstant now) {
        { // protect only atomics
            TGuard g(Spinlock_);

            TStateDetails sd = State_.load(std::memory_order_relaxed);
            if (
                sd.Is(toSet) ||                           // to keep lastChangeTime actual
                skipIfExpectedMathched == sd.Is(expected) // to process ForceDown
            ) {
                return;
            }

            sd.Make(toSet, now);
            State_.store(sd, std::memory_order_relaxed);
        }

        if (LogLevel_ == ELogLevel::Verbose) {
            Log_.Debug() << Dsn_ << " state=" << toSet;
        }
    }

    TPoolState::EState TPoolState::CheckState(TDuration& lastChangeTime, TInstant now) const {
        const TStateDetails sd = State_.load(std::memory_order_relaxed);
        lastChangeTime = now - sd.GetLastChangingTime();

        // - if host provides status 'unavailable'
        // - if pool is empty for a "long" time signal unavailability
        const EState newState = sd.Is(ForceDown)
                                    ? ForceDown
                                    : (sd.Is(Down) && lastChangeTime > FailThreshold_ ? Down : Up);

        const EState oldState = PrevState_.exchange(newState);

        if (newState != oldState) {
            Log_.Warning() << Dsn_ << " State switched from " << oldState << " to " << newState;
        }

        return newState;
    }

    TPoolState::EState TPoolState::GetState() const {
        return State_.load(std::memory_order_relaxed).Get();
    }

    bool TPoolState::IsOk(const TPoolState& state, const TDbInfo& dbInfo, TString* statusBuf) {
        if (statusBuf) {
            statusBuf->clear();
        }

        TDuration lastChangeTime;
        switch (state.CheckState(lastChangeTime)) {
            case TPoolState::Up:
                if (statusBuf) {
                    NUtils::Append(*statusBuf, dbInfo.Serialized, " is OK");
                }
                return true;
            case TPoolState::Down:
                if (statusBuf) {
                    TStringOutput o(*statusBuf);
                    o << dbInfo.Serialized << " is unavailable for " << HumanReadable(lastChangeTime);
                }
                return false;
            case TPoolState::ForceDown:
                if (statusBuf) {
                    TStringOutput o(*statusBuf);
                    o << dbInfo.Serialized << " is forced down for " << HumanReadable(lastChangeTime);
                }
                return false;
        }
    }

    static const ui64 onlyTimeMask = 0x00FFFFFFFFFFFFFF;
    static const int stateOffset = 56;

    TInstant TPoolState::TStateDetails::GetLastChangingTime() const {
        return TInstant::MilliSeconds(Val_ & onlyTimeMask);
    }

    TPoolState::EState TPoolState::TStateDetails::Get() const {
        return EState(Val_ >> stateOffset);
    }

    bool TPoolState::TStateDetails::Is(EState state) const {
        return (Val_ >> stateOffset) == (ui64)state;
    }

    static ui64 CreateMask(TPoolState::EState state) {
        return ((ui64)state) << stateOffset;
    }

    void TPoolState::TStateDetails::Make(EState state, TInstant now) {
        Val_ = now.MilliSeconds() | CreateMask(state);
    }
}
