#include "producer.h"

#include "handle.h"
#include "host.h"

#include <passport/infra/libs/cpp/dbpool/exception.h>

#include <util/stream/format.h>

namespace NPassport::NDbPool {
    TProducableHost::TProducableHost(TDbPoolLog log,
                                     TDuration connectionTimeout,
                                     double totalSizeRatio,
                                     std::reference_wrapper<THost> host)
        : Host_(host)
        , Log_(log)
        , ConnectionTimeout_(connectionTimeout)
        , TotalSizeRatio_(totalSizeRatio)
        , DbInfo_(Host_.get().GetDbInfo())
    {
    }

    TDuration TProducableHost::ProduceNonblocking(const TManualEvent& wakeUpServiceThread) {
        if (Handle_) {
            // Finish creation process from previous iteration
            if (ECheckStatus::Failed == CheckHandleInProcess()) {
                Log_.Debug()
                    << DbInfo_
                    << " Producer: can't connect for " << HumanReadable(TInstant::Now() - FirstFailed_)
                    << ". id=" << DbInfo_.DisplayName;
            }
        }

        TDuration toSleep = TDuration::Max();

        if (Handle_) {
            toSleep = GetHandleDeadline() - TInstant::Now();
        } else if ((IsOk_ || TryNext_ < TInstant::Now()) && NeedMoreHandles()) {
            if (!IsOk_) {
                // Let backend to catch his breath
                TryNext_ = TInstant::Now() + ConnectionTimeout_;
            }

            CreateNewHandle(wakeUpServiceThread);
            toSleep = ConnectionTimeout_;
        }

        CheckBadHandle();

        return toSleep;
    }

    void TProducableHost::Wait() {
        if (Handle_) {
            // Finish creation process
            Handle_->InitionError.Wait(GetHandleDeadline());
        }
    }

    TProducableHost::ECheckStatus TProducableHost::CheckHandleInProcess() {
        Y_VERIFY(Handle_);

        try {
            if (!IsHandleReady()) {
                return ECheckStatus::InProgress;
            }

            const TString randid = std::move(Handle_->Randid);
            Host_.get().PutNewHandle(std::move(Handle_->Handle));
            IsOk_ = true;
            Handle_.reset();

            Log_.Debug() << DbInfo_
                         << " Producer: creating handle OK (randid=" << randid
                         << ") id=" << DbInfo_.DisplayName;
            return ECheckStatus::Ok;
        } catch (const std::exception& e) {
            if (IsOk_) {
                IsOk_ = false;
                FirstFailed_ = TInstant::Now();
            }

            Handle_.reset();
            Log_.Debug() << DbInfo_
                         << " Producer: error on checking new handle <" << e.what()
                         << "> id=" << DbInfo_.DisplayName;
        }

        return ECheckStatus::Failed;
    }

    bool TProducableHost::IsHandleReady() const {
        Y_VERIFY(Handle_);

        if (!Handle_->InitionError.HasValue()) {
            if (TInstant::Now() < GetHandleDeadline()) {
                return false;
            }

            throw TTimeoutException(Handle_->Handle->GetDbInfo())
                << "Producer: connection timeout ("
                << HumanReadable(ConnectionTimeout_)
                << ") randid=" << Handle_->Randid;
        }

        const THandleInitError& err = Handle_->InitionError.GetValue();
        if (err) {
            throw TTimeoutException(Handle_->Handle->GetDbInfo())
                << " Producer: connection failed: " << *err
                << " (randid=" << Handle_->Randid << ")";
        }

        return true;
    }

    bool TProducableHost::NeedMoreHandles() const {
        THandlesInfo info = Host_.get().GetHandlesInfo();

        if (info.ConfigCount <= info.Good) {
            return false;
        }

        size_t totalLimit = TotalSizeRatio_ * info.ConfigCount;
        if (totalLimit <= info.Total) {
            Log_.Warning() << DbInfo_ << " Total worker count was reached: " << info.Total
                           << " of " << totalLimit;
            return false;
        }

        return true;
    }

    void TProducableHost::CreateNewHandle(const TManualEvent& wakeUpServiceThread) {
        Y_VERIFY(!Handle_);

        try {
            Log_.Debug() << DbInfo_ << " Producer: creating handle. id=" << DbInfo_.DisplayName;

            Handle_ = Host_.get().CreateHandle();
            Handle_->InitionError.Subscribe(
                [ev = wakeUpServiceThread](const auto&) mutable {
                    ev.Signal();
                });
        } catch (const std::exception& e) {
            Log_.Debug() << DbInfo_
                         << " Producer: error <" << e.what()
                         << "> id=" << DbInfo_.DisplayName;
        }
    }

    void TProducableHost::CheckBadHandle() {
        try {
            Host_.get().CleanupOneBad();
        } catch (const std::exception& e) {
            // Impossible case
            Log_.Debug() << DbInfo_
                         << " Producer: error on checking bad handle <" << e.what()
                         << "> id=" << DbInfo_.DisplayName;
        }
    }

    TInstant TProducableHost::GetHandleDeadline() const noexcept {
        Y_VERIFY(Handle_);
        return Handle_->Start + ConnectionTimeout_;
    }

    TProducer::TProducer(TDbPoolLog log,
                         const TDbPoolSettings& settings,
                         const THostRefs& hosts)
        : Log_(log)
    {
        for (const std::reference_wrapper<THost>& h : hosts) {
            Hosts_.push_back(TProducableHost(
                Log_,
                settings.ConnectionTimeout,
                settings.TotalSizeRatio,
                h));
        }
    }

    TProducer::~TProducer() {
        try {
            WaitAll();
        } catch (...) {
        }
    }

    TDuration TProducer::ProduceNonblocking(const TManualEvent& wakeUpServiceThread) {
        TDuration toSleep = TDuration::Max();

        for (TProducableHost& h : Hosts_) {
            try {
                TDuration s = h.ProduceNonblocking(wakeUpServiceThread);
                toSleep = std::min(toSleep, s);
            } catch (const std::exception& e) {
                // Impossible case
                Log_.Debug() << h.DbInfo()
                             << " Producer: error on checking previous iteration <" << e.what()
                             << "> id=" << h.DbInfo().DisplayName;
            }
        }

        return toSleep;
    }

    void TProducer::WaitAll() {
        for (TProducableHost& h : Hosts_) {
            try {
                h.Wait();
            } catch (const std::exception& e) {
                Log_.Debug() << h.DbInfo()
                             << " Producer: error on wating <" << e.what()
                             << "> id=" << h.DbInfo().DisplayName;
            }
        }
    }
}
