#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/dbpool/db_pool_ctx.h>

#include <passport/infra/libs/cpp/json/config.h>
#include <passport/infra/libs/cpp/utils/thread_local_id.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/file.h>
#include <util/system/fs.h>

using namespace NPassport;
using namespace NPassport::NDbPool;

Y_UNIT_TEST_SUITE(PoolCtx) {
    static const TString configFile = ArcadiaSourceRoot() + "/passport/infra/libs/cpp/dbpool/ut/config.xml";
    const TString tmpFileName("tmp_log_file");

    Y_UNIT_TEST(createDBXml) {
        std::shared_ptr<TDbPoolCtx> ctx(new TDbPoolCtx);

        NXml::TConfig conf = NXml::TConfig::ReadFromFile(configFile);

        std::shared_ptr<NDbPool::TDbPool> db;
        UNIT_ASSERT_EXCEPTION_CONTAINS(db = ctx->CreateDb(conf, "/root/("), yexception, "Invalid expression");
        UNIT_ASSERT_EXCEPTION_CONTAINS(db = ctx->CreateDb(conf, "/root"), yexception, "nonexistent config param: /root/db_driver");
        UNIT_ASSERT_NO_EXCEPTION(db = ctx->CreateDb(conf, "/root/default_db"));
        UNIT_ASSERT_VALUES_EQUAL(4, db->GetHandlesInfo().ConfigCount);
    }

    Y_UNIT_TEST(createDBJson) {
        std::shared_ptr<TDbPoolCtx> ctx = TDbPoolCtx::Create();

        NJson::TConfig conf = NJson::TConfig::ReadFromMemory(
            R"(
            {
                "foo": {},
                "default_db": {
                    "poolsize": 4,
                    "get_timeout": 30,
                    "connect_timeout": 300,
                    "query_timeout": 50,
                    "db_driver": "sqlite",
                    "db_port": 7,
                    "db_name": "db/safeguarddb.sqlite3.db",
                    "db_host": [
                        {"host": "default_db_host"}
                    ]
                 }
            })");

        std::shared_ptr<NDbPool::TDbPool> db;
        UNIT_ASSERT_EXCEPTION_CONTAINS(db = ctx->CreateDb(conf, "/foo/("), yexception, "jsonpoint not found: /foo/(/db_driver");
        UNIT_ASSERT_EXCEPTION_CONTAINS(db = ctx->CreateDb(conf, "/foo"), yexception, "jsonpoint not found: /foo/db_driver");
        UNIT_ASSERT_NO_EXCEPTION(db = ctx->CreateDb(conf, "/default_db"));
        UNIT_ASSERT_VALUES_EQUAL(4, db->GetHandlesInfo().ConfigCount);
    }

    Y_UNIT_TEST(credentialsReadFromFile) {
        struct TTestCase {
            TString File;
            TString Error;
        };

        std::vector<TTestCase> cases = {
            TTestCase{
                .File = R"()",
                .Error = "Failed to parse json object",
            },
            TTestCase{
                .File = R"([])",
                .Error = "Failed to parse json object",
            },
            TTestCase{
                .File = R"({})",
                .Error = "jsonpoint not found: /db_user",
            },
            TTestCase{
                .File = R"({"db_user":""})",
                .Error = "value is empty: /db_user",
            },
            TTestCase{
                .File = R"({"db_user":"foo"})",
                .Error = "",
            },
            TTestCase{
                .File = R"({"db_user":"foo","db_pass":""})",
                .Error = "",
            },
        };

        const TString tmpFile = "./tmp.creds";
        UNIT_ASSERT_EXCEPTION(
            TDbPoolCtx::TCredentials::ReadFromFile(tmpFile),
            yexception);

        for (const TTestCase& c : cases) {
            NFs::Remove(tmpFile);
            TFileOutput(tmpFile).Write(c.File);

            if (c.Error) {
                UNIT_ASSERT_EXCEPTION_CONTAINS_C(
                    TDbPoolCtx::TCredentials::ReadFromFile(tmpFile),
                    yexception,
                    c.Error,
                    c.Error);
            } else {
                UNIT_ASSERT_NO_EXCEPTION(
                    TDbPoolCtx::TCredentials::ReadFromFile(tmpFile));
            }
        }

        NFs::Remove(tmpFile);
        TFileOutput(tmpFile).Write(R"({"db_user":"foo","db_pass":"bar"})");

        TDbPoolCtx::TCredentials cred;
        UNIT_ASSERT_NO_EXCEPTION(
            cred = TDbPoolCtx::TCredentials::ReadFromFile(tmpFile));

        UNIT_ASSERT_VALUES_EQUAL("foo", cred.User);
        UNIT_ASSERT_VALUES_EQUAL("bar", cred.Password);
    }
}
