#include <passport/infra/libs/cpp/dbpool/misc/poolstate.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NDbPool;

Y_UNIT_TEST_SUITE(PoolImpl) {
    Y_UNIT_TEST(StateDetails) {
        TPoolState::TStateDetails sd;

        UNIT_ASSERT(sd.Is(TPoolState::Up));
        UNIT_ASSERT_VALUES_EQUAL(TInstant(), sd.GetLastChangingTime());

        for (TPoolState::EState marker : {
                 TPoolState::Up,
                 TPoolState::Down,
                 TPoolState::ForceDown,
                 TPoolState::Up,
                 TPoolState::Up,
                 TPoolState::Down,
                 TPoolState::Down,
                 TPoolState::ForceDown,
                 TPoolState::ForceDown,
             }) {
            TInstant now = TInstant::Now();
            sd.Make(marker, now);

            for (TPoolState::EState other : {
                     TPoolState::Up,
                     TPoolState::Down,
                     TPoolState::ForceDown,
                 }) {
                if (other == marker) {
                    UNIT_ASSERT_C(sd.Is(marker), marker);
                } else {
                    UNIT_ASSERT_C(!sd.Is(other), marker << " vs " << other);
                }
            }
            UNIT_ASSERT_VALUES_EQUAL_C(TInstant::MilliSeconds(now.MilliSeconds()), sd.GetLastChangingTime(), marker);
        }
    }

    Y_UNIT_TEST(PoolState_tryMake) {
        TPoolState s(TDbPoolLog(), TDuration::Minutes(1), "some dsn");

        UNIT_ASSERT_VALUES_EQUAL(TPoolState::Up, s.GetState());

        for (TPoolState::EState state : {
                 TPoolState::Up,
                 TPoolState::Down,
                 TPoolState::ForceDown,
                 TPoolState::Up,
                 TPoolState::Up,
                 TPoolState::Down,
                 TPoolState::Down,
                 TPoolState::ForceDown,
                 TPoolState::ForceDown,
             }) {
            s.TryMake(state);
            UNIT_ASSERT_VALUES_EQUAL(state, s.GetState());
        }
    }

    Y_UNIT_TEST(PoolState_tryMakeIfMatch) {
        for (TPoolState::EState originalState : {
                 TPoolState::Up,
                 TPoolState::Down,
                 TPoolState::ForceDown,
             }) {
            for (TPoolState::EState targetState : {
                     TPoolState::Up,
                     TPoolState::Down,
                     TPoolState::ForceDown,
                 }) {
                for (TPoolState::EState expected : {
                         TPoolState::Up,
                         TPoolState::Down,
                         TPoolState::ForceDown,
                     }) {
                    TPoolState s(TDbPoolLog(), TDuration::Minutes(1), "some dsn");
                    s.TryMake(originalState);
                    s.TryMakeIfMatch(targetState, expected);

                    if (originalState == expected) {
                        UNIT_ASSERT_VALUES_EQUAL_C(
                            targetState,
                            s.GetState(),
                            "originalState:" << originalState << ",targetState:" << targetState << ",expected:" << expected);
                    } else {
                        UNIT_ASSERT_VALUES_EQUAL_C(
                            originalState,
                            s.GetState(),
                            "originalState:" << originalState << ",targetState:" << targetState << ",expected:" << expected);
                    }
                }
            }
        }
    }

    Y_UNIT_TEST(PoolState_tryMakeIfNotMatch) {
        for (TPoolState::EState originalState : {
                 TPoolState::Up,
                 TPoolState::Down,
                 TPoolState::ForceDown,
             }) {
            for (TPoolState::EState targetState : {
                     TPoolState::Up,
                     TPoolState::Down,
                     TPoolState::ForceDown,
                 }) {
                for (TPoolState::EState expected : {
                         TPoolState::Up,
                         TPoolState::Down,
                         TPoolState::ForceDown,
                     }) {
                    TPoolState s(TDbPoolLog(), TDuration::Minutes(1), "some dsn");
                    s.TryMake(originalState);
                    s.TryMakeIfNotMatch(targetState, expected);

                    if (originalState != expected) {
                        UNIT_ASSERT_VALUES_EQUAL_C(
                            targetState,
                            s.GetState(),
                            "originalState:" << originalState << ",targetState:" << targetState << ",expected:" << expected);
                    } else {
                        UNIT_ASSERT_VALUES_EQUAL_C(
                            originalState,
                            s.GetState(),
                            "originalState:" << originalState << ",targetState:" << targetState << ",expected:" << expected);
                    }
                }
            }
        }
    }

    Y_UNIT_TEST(PoolState_checkState) {
        TPoolState s(TDbPoolLog(), TDuration::Minutes(1), "some dsn");
        TInstant now = TInstant::Now();
        now -= TDuration::MicroSeconds(now.MicroSecondsOfSecond()); // state does not contain microseconds

        TDuration lastChangeTime;

        UNIT_ASSERT_VALUES_EQUAL(TPoolState::Up, s.GetState());
        UNIT_ASSERT_VALUES_EQUAL(TPoolState::Up, s.CheckState(lastChangeTime, now));
        UNIT_ASSERT_VALUES_EQUAL(TPoolState::Up, s.CheckState(lastChangeTime, now + TDuration::Minutes(2)));

        s.TryMake(TPoolState::Down, now);

        UNIT_ASSERT_VALUES_EQUAL(TPoolState::Down, s.GetState());
        UNIT_ASSERT_VALUES_EQUAL(TPoolState::Up, s.CheckState(lastChangeTime, now));
        UNIT_ASSERT_VALUES_EQUAL(TPoolState::Down, s.CheckState(lastChangeTime, now + TDuration::Minutes(2)));
        UNIT_ASSERT_VALUES_EQUAL(TDuration::Minutes(2), lastChangeTime);

        UNIT_ASSERT_VALUES_EQUAL(TPoolState::Down, s.GetState());
        UNIT_ASSERT_VALUES_EQUAL(TPoolState::Down, s.CheckState(lastChangeTime, now + TDuration::Minutes(5)));
        UNIT_ASSERT_VALUES_EQUAL(TDuration::Minutes(5), lastChangeTime);

        s.TryMakeIfNotMatch(TPoolState::Down, TPoolState::ForceDown, now);

        UNIT_ASSERT_VALUES_EQUAL(TPoolState::Down, s.GetState());
        UNIT_ASSERT_VALUES_EQUAL(TPoolState::Up, s.CheckState(lastChangeTime, now));
        UNIT_ASSERT_VALUES_EQUAL(TPoolState::Down, s.CheckState(lastChangeTime, now + TDuration::Minutes(2)));
        UNIT_ASSERT_VALUES_EQUAL(TDuration::Minutes(2), lastChangeTime);

        UNIT_ASSERT_VALUES_EQUAL(TPoolState::Down, s.GetState());
        UNIT_ASSERT_VALUES_EQUAL(TPoolState::Down, s.CheckState(lastChangeTime, now + TDuration::Minutes(5)));
        UNIT_ASSERT_VALUES_EQUAL(TDuration::Minutes(5), lastChangeTime);

        s.TryMake(TPoolState::Up, now);

        UNIT_ASSERT_VALUES_EQUAL(TPoolState::Up, s.GetState());
        UNIT_ASSERT_VALUES_EQUAL(TPoolState::Up, s.CheckState(lastChangeTime, now + TDuration::Minutes(10)));

        s.TryMake(TPoolState::ForceDown, now);

        UNIT_ASSERT_VALUES_EQUAL(TPoolState::ForceDown, s.GetState());
        UNIT_ASSERT_VALUES_EQUAL(TPoolState::ForceDown, s.CheckState(lastChangeTime, now));
    }
}
