#include "util.h"

#include "handle.h"

#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/string/cast.h>

#include <errno.h>
#include <vector>

namespace NPassport::NDbPool::NUtils {
    TResponseWithRetries DoQueryTries(NDbPool::TDbPool& db,
                                      const TQuery& query,
                                      ui16 tries) {
        size_t retries = 0;
        const TDuration sleepPeriod = TDuration::Seconds(1);

        while (tries--) {
            const TInstant start = TInstant::Now();

            try {
                NDbPool::TBlockingHandle sqlh(db);
                return {
                    .Result = sqlh.Query(query.Clone()),
                    .LastResponseTime = TInstant::Now() - start,
                    .Retries = retries,
                };
            } catch (const NDbPool::TException& e) {
                TLog::Debug() << "Request failed: " << query.Query()
                              << ". Retries left: " << tries
                              << ". Took: " << (TInstant::Now() - start)
                              << ". Reason: " << e.what()
                              << ". Sleep for " << sleepPeriod;
            }

            // Let backend to catch his breath
            Sleep(sleepPeriod);
            ++retries;
        }

        ythrow NDbPool::TException(db.GetDbInfo())
            << "Failed to make reliable fetching."
            << " Query: " << query.Query();
    }

    TString GetHttpBody(TDbPool& db,
                        TQuery&& query,
                        const char* errMsg) {
        TString output;
        TString err;
        const TString path = query.Query();

        NDbPool::TBlockingHandle h(db);
        if (!FetchBodyFromHttpResult(h.Query(std::move(query)),
                                     path,
                                     output,
                                     err))
        {
            throw TException(h.GetDbInfo()) << errMsg << ". " << err << ". " << output;
        }

        return output;
    }

    bool FetchBodyFromHttpResult(std::unique_ptr<TResult> result,
                                 const TStringBuf path,
                                 TString& output,
                                 TString& errMsg) {
        output.clear();
        errMsg.clear();

        THttpResponse response;
        try {
            response = result->ToHttpResponse();
        } catch (const TFormatException& e) {
            NPassport::NUtils::AppendExt(
                errMsg, 256,
                "failed to fetch response.",
                " path='", path, '\'',
                ". Error: ", e.what());
            return false;
        }

        output = std::move(response.Body);

        if (response.Status != 200) {
            NPassport::NUtils::AppendExt(
                errMsg, 256,
                "returned http status=", response.Status,
                " path='", path, '\'');
            return false;
        }

        return true;
    }
}
