#pragma once

#include "exception.h"

#include <util/generic/string.h>
#include <util/string/cast.h>

#include <optional>
#include <vector>

namespace NPassport::NDbPool {
    class TValue: TMoveOnly {
    public:
        TValue(TString&& v, bool isNull = false, bool isBlob = false)
            : Val_(std::move(v))
            , IsNull_(isNull)
            , IsBlob_(isBlob)
        {
        }

        TValue() = default;

        int AsInt() const {
            return AsNum<int>();
        }

        template <typename T>
        auto As() const {
            if constexpr (std::is_integral_v<std::remove_reference_t<T>>) {
                return AsNum<T>();
            } else {
                return AsString();
            }
        }

        const TString& AsString() const;
        operator const TString&() const {
            return AsString();
        }

        TString& AsMutableString();

        bool IsNull() const {
            return IsNull_;
        }

    private:
        template <typename T>
        typename std::remove_reference_t<T> AsNum() const {
            using TActualType = std::remove_reference_t<T>;
            static_assert(std::is_integral<TActualType>::value, "Allow only interger types");

            if (IsNull_) {
                throw TNilException();
            }

            if (IsBlob_) {
                throw TFormatException() << "The Value is blob";
            }

            TActualType res = 0;
            if (!TryIntFromString<10>(Val_, res)) {
                throw TFormatException() << "invalid integer format: '" << Val_ << "'";
            }

            return res;
        }

    private:
        TString Val_;
        bool IsNull_ = true;
        bool IsBlob_ = false;
    };

    struct TRow: TMoveOnly, std::vector<TValue> {
        template <class... Types>
        void Fetch(Types&... args) const {
            const size_t expectedArgs = sizeof...(args);
            const size_t actualArgs = size();

            Y_ENSURE(expectedArgs == actualArgs,
                     "expectedArgs==" << expectedArgs << ". actualArgs==" << actualArgs);

            FetchNext(args...);
        }

        TString DebugString() const;

    private:
        template <size_t Idx = 0, typename T, typename... Types>
        void FetchNext(T& arg, Types&... args) const {
            FetchNext<Idx>(arg);
            FetchNext<Idx + 1>(args...);
        }

        template <typename T>
        struct TIsOptional {
            static constexpr bool value = false;
        };

        template <typename T>
        struct TIsOptional<std::optional<T>> {
            static constexpr bool value = true;
        };

        template <size_t Idx = 0, typename T, typename... Types>
        void FetchNext(T& arg) const {
            if constexpr (TIsOptional<T>::value) {
                const TValue& v = at(Idx);

                if (v.IsNull()) {
                    arg.reset();
                } else {
                    using TActualType = typename T::value_type;
                    arg = v.As<TActualType>();
                }
            } else {
                arg = at(Idx).As<T>();
            }
        }
    };

    struct TTable: TMoveOnly, std::vector<TRow> {
    };
}
