#include "db_pool_ctx.h"

#include "db_pool.h"

#include <passport/infra/libs/cpp/dbpool/misc/counter.h>
#include <passport/infra/libs/cpp/unistat/time_stat.h>
#include <passport/infra/libs/cpp/xml/config.h>

namespace NPassport::NDbPool2 {
    TDbPoolCtxPtr TDbPoolCtx::Create() {
        return std::make_shared<TDbPoolCtx>();
    }

    void TDbPoolCtx::InitLogger(const TString& fileName) {
        if (fileName.empty() || fileName == "_NOLOG_") {
            return;
        }

        Log_ = TDbPoolLog(fileName);
    }

    void TDbPoolCtx::InitLogger(TDbPoolLog log) {
        Log_ = log;
    }

    TDbPoolLog TDbPoolCtx::GetLogger() const {
        return Log_;
    }

    std::unique_ptr<TDbPool> TDbPoolCtx::CreateDb(const NXml::TConfig& config, const TString& xpath) {
        return CreateDb(config, xpath, xpath);
    }

    std::unique_ptr<TDbPool> TDbPoolCtx::CreateDb(const NXml::TConfig& config,
                                                  const TString& xpath_db,
                                                  const TString& xpath_settings) {
        size_t dbpoolsize = config.AsInt(xpath_settings + "/poolsize", 10);
        int get_tm = config.AsInt(xpath_settings + "/get_timeout", 10);
        int cn_tm = config.AsInt(xpath_settings + "/connect_timeout", 500);
        int q_tm = config.AsInt(xpath_settings + "/query_timeout", 500);
        int f_tm = config.AsInt(xpath_settings + "/fail_threshold", 500);
        int pingPeriod = config.AsInt(xpath_settings + "/ping_period", 3000);
        int timeToInit = config.AsInt(xpath_settings + "/time_to_init", 2000);

        TString driver = config.AsString(xpath_db + "/db_driver");
        ui16 port = config.AsNum<ui16>(xpath_db + "/db_port", 0);
        TString dbname = config.AsString(xpath_db + "/db_name");

        NDbPool::TDbPoolCtx::TCredentials creds;
        if (TString path = xpath_db + "/db_credentials"; config.Contains(path)) {
            // throws on empty db_user, but allows empty db_pass
            creds = NDbPool::TDbPoolCtx::TCredentials::ReadFromFile(config.AsString(path));
        } else {
            creds.User = config.AsString(xpath_db + "/db_user", "");
            creds.Password = config.AsString(xpath_db + "/db_pass", "");
        }
        TString displayName = config.AsString(xpath_settings + "/display_name", dbname);

        TString locale_cmd = config.AsString(xpath_settings + "/locale_cmd", "");

        std::vector<NDbPool::TDbHost> hosts;
        for (const TString& path : config.SubKeys(xpath_db + "/db_host")) {
            hosts.push_back(NDbPool::TDbHost{
                .Host = config.AsString(path),
                .Port = port,
                .Weight = config.AsNum<size_t>(path + "/@weight", 1),
            });
        }

        if (hosts.empty()) {
            throw yexception() << "db_host is missing at " << xpath_db;
        }

        NDbPool::TDestination::TExtendedParams ext;
        for (const TString& path : config.SubKeys(xpath_db + "/extended")) {
            ext.insert({config.AsString(path + "/@key"), config.AsString(path)});
        }

        NDbPool::TDbPoolSettings settings{
            .Dsn = NDbPool::TDestination::Create(
                driver,
                creds.User,
                creds.Password,
                dbname,
                locale_cmd,
                displayName,
                std::move(ext)),
            .Hosts = std::move(hosts),
            .Size = dbpoolsize,
            .GetTimeout = TDuration::MilliSeconds(get_tm),
            .ConnectionTimeout = TDuration::MilliSeconds(cn_tm),
            .QueryTimeout = TDuration::MilliSeconds(q_tm),
            .FailThreshold = TDuration::MilliSeconds(f_tm),
            .PingPeriod = TDuration::MilliSeconds(pingPeriod),
            .TimeToInit = TDuration::MilliSeconds(timeToInit),
        };

        return std::make_unique<TDbPool>(settings, shared_from_this());
    }

    void TDbPoolCtx::AddDbpool(TDbPoolCtx::TTimeStat st, TDbPoolCtx::TCounters c) {
        TimeStats_.push_back(std::move(st));
        Counters_.push_back(std::move(c));
    }

    void TDbPoolCtx::AddUnistat(NUnistat::TBuilder& builder) {
        for (const TTimeStat& t : TimeStats_) {
            t->AddUnistat(builder);
        }
        for (const TCounters& t : Counters_) {
            t->AddUnistat(builder);
        }
    }
}
