#include "handle.h"

#include "db_pool.h"
#include "poller.h"
#include "misc/utils.h"

#include <passport/infra/libs/cpp/dbpool/exception.h>
#include <passport/infra/libs/cpp/unistat/time_stat.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

#include <util/thread/singleton.h>

namespace NPassport::NDbPool2 {
    THandle::THandle()
        : State_(EHandleState::Destroyed)
    {
    }

    THandle::THandle(TDbPool& dbpool)
        : THandle(dbpool, GetThreadLocalPoller())
    {
    }

    THandle::THandle(TDbPool& dbpool, TPoller& poller)
        : Dbpool_(&dbpool)
        , Poller_(&poller)
        , Driver_(Dbpool_->GetDriver())
    {
    }

    THandle::~THandle() {
        try {
            if (Driver_) {
                Poller_->Forget(Driver_);
                if (EHandleState::WaitingResult == State_) {
                    Dbpool_->PutDriverWithTimeOut(std::move(Driver_));
                } else {
                    Dbpool_->PutDriverWithError(std::move(Driver_));
                }
            }
        } catch (...) {
            // ¯\_(ツ)_/¯
        }
    }

    THandle::THandle(THandle&&) noexcept = default;

    THandle& THandle::operator=(THandle&&) noexcept = default;

    TString THandle::EscapeQueryParam(const TString& str) const {
        Y_ENSURE(Driver_, "driver is null");
        return Driver_->EscapeQueryParam(str);
    }

    THandle& THandle::SendQuery(NDbPool::TQuery&& query) {
        Y_ENSURE(EHandleState::ReadyForQuery == State_, "handle cannot send query: " << State_);

        Dbpool_->IncAllRequests();     // for unistat
        QueryStart_ = TInstant::Now(); // for processing timeout and for unistat

        Future_ = Driver_->StartSendingQuery(std::move(query));
        ProcessResponseTime();

        State_ = EHandleState::WaitingResult;
        Poller_->Add(Driver_);

        return *this;
    }

    NDbPool::TTable THandle::ExtractResult(TDuration customTimeout) {
        Y_ENSURE(EHandleState::WaitingResult == State_, "handle cannot extract result: " << State_);
        State_ = EHandleState::Destroyed;

        const TDuration timeout = TDuration() == customTimeout
                                      ? Dbpool_->GetQueryTimeout()
                                      : customTimeout;

        if (!Future_.HasValue()) {
            // wait response for this driver and may be for some other driver too
            const TInstant deadline = QueryStart_ + timeout;
            while (TInstant::Now() < deadline && !Future_.HasValue()) {
                Poller_->RunLoopOnce(deadline);
            }
        }

        Poller_->Forget(Driver_); // do not burn CPU any more for this driver

        if (!Future_.HasValue()) {
            const TDriverDestination dst = Driver_->GetDestination();
            Dbpool_->PutDriverWithTimeOut(std::move(Driver_));
            throw NDbPool::TTimeoutException(dst.Serialized)
                << "query timeout (" << timeout << ")";
        }

        // got result from backend
        IDriver::TErrorOr<NDbPool::TTable> resultOrError = Future_.ExtractValue();
        ThrowOnError(resultOrError, Dbpool_->GetLogger(), [this]() {
            Dbpool_->PutDriverWithError(std::move(Driver_));
        });

        Dbpool_->PutGoodDriver(std::move(Driver_));

        return std::move(std::get<NDbPool::TTable>(resultOrError));
    }

    void THandle::ProcessResponseTime() {
        TDbPoolCtx::TTimeStat stats = Dbpool_->GetTimeStats();
        TInstant start = QueryStart_;

        // this callback will be called after value setting.
        // should store copies of objects instead of references to avoid access to deleted objects
        Future_.Subscribe([stats = std::move(stats), start = start](const IDriver::TResultType&) {
            stats->Insert(TInstant::Now() - start);
            // TODO: log long queries
        });
    }
}
