#include "db_pool_impl.h"

#include "driver_creator.h"
#include "driver_pinger.h"

#include <passport/infra/libs/cpp/dbpool2/misc/utils.h>

#include <passport/infra/libs/cpp/dbpool2/db_pool_ctx.h>
#include <passport/infra/libs/cpp/dbpool2/poller.h>

#include <passport/infra/libs/cpp/unistat/time_stat.h>

namespace NPassport::NDbPool2 {
    TDbPoolImpl::TDbPoolImpl(const NDbPool::TDbPoolSettings& settings, std::shared_ptr<TDbPoolCtx> ctx)
        : Settings_(settings)
        , Destination_(CreateDriverDestination(Settings_))
        , SizeLimit_(CreateSizeLimit(Settings_))
        , Log_(ctx->GetLogger())
        , TimeStats_(CreateTimeStats(Settings_))
        , Counters_(std::make_shared<NDbPool::TCounters>(Settings_.Dsn->DisplayName, Settings_.Size))
        , DbInfo_(Destination_.Serialized)
        , State_(Log_, Settings_.FailThreshold, "")
    {
        DbInfo_.Db = Settings_.Dsn->DisplayName;
        ctx->AddDbpool(TimeStats_, Counters_);
    }

    void TDbPoolImpl::Stop() {
        Stopping_ = true;
        Log_.Info("%s: stopping pool", GetDsn());
    }

    bool TDbPoolImpl::IsOk(TString* status) const {
        TDuration failSpan;
        switch (State_.CheckState(failSpan)) {
            case NDbPool::TPoolState::Up:
                if (status) {
                    *status = NUtils::CreateStr(Destination_.Serialized, " is OK");
                }
                return true;
            case NDbPool::TPoolState::Down:
                if (status) {
                    *status = NUtils::CreateStr(Destination_.Serialized,
                                                " is unavailable for ",
                                                failSpan.MilliSeconds(),
                                                " ms");
                }
                return false;
            case NDbPool::TPoolState::ForceDown:
                if (status) {
                    *status = NUtils::CreateStr(Destination_.Serialized,
                                                " is forced down for ",
                                                failSpan.MilliSeconds(),
                                                " ms");
                }
                return false;
        }
    }

    void TDbPoolImpl::TryPing() {
        TDriverPtr driver = GetDriver();
        Log_.Debug("%s: dr=%lu: tryPing: starting", GetDsn(), driver->GetId());

        IDriver::TResultFuture<bool> future = driver->StartPinging();

        TPoller poller;
        poller.Add(driver);

        const TInstant queryStart = TInstant::Now();
        if (!future.HasValue()) {
            const TInstant deadline = queryStart + Settings_.QueryTimeout;
            while (TInstant::Now() < deadline && !future.HasValue()) {
                poller.RunLoopOnce(deadline);
            }
        }

        if (!future.HasValue()) {
            PutDriverWithTimeOut(driver);
            throw yexception() << "Could not initialize connection to " << Destination_.Serialized;
        }

        ThrowOnError(future.GetValue(), Log_, [&, this]() {
            PutDriverWithError(driver);
        });

        Log_.Debug("%s: dr=%lu: tryPing: success", GetDsn(), driver->GetId());
        PutGoodDriver(std::move(driver));
    }

    NDbPool::THandlesInfo TDbPoolImpl::GetHandlesInfo() const {
        return Counters_->GetHandlesInfo();
    }

    const NDbPool::TDbInfo& TDbPoolImpl::GetDbInfo() const {
        return DbInfo_;
    }

    TDbPoolImpl::TDriverPtr TDbPoolImpl::GetDriver() {
        const TInstant start = TInstant::Now();
        TDriverPtr driver;

        while (!IdleDrivers_.Dequeue(&driver) && TInstant::Now() - start < Settings_.GetTimeout) {
            // there is no one idle driver
            Sleep(TDuration::MilliSeconds(1));
        }

        if (driver) {
            --Counters_->IdleHandles;
            return driver;
        }

        State_.TryMake(NDbPool::TPoolState::Down);
        ++Counters_->CantGetConnection;

        throw NDbPool::TCantGetConnection(Destination_.Serialized) << "Can't get connection";
    }

    void TDbPoolImpl::PutGoodDriver(TDbPoolImpl::TDriverPtr driver) {
        Y_ENSURE(driver, "got null driver: " << GetDsn());

        IdleDrivers_.Enqueue(std::move(driver));
        ++Counters_->IdleHandles;
        if (Counters_->GoodHandles.GetValue() >= Settings_.Size) {
            State_.TryMake(NDbPool::TPoolState::Up);
        }
    }

    void TDbPoolImpl::PutDriverWithError(TDbPoolImpl::TDriverPtr driver) {
        Y_ENSURE(driver, "got null driver: " << GetDsn());

        Counters_->TotalHandles.Dec();
        --Counters_->GoodHandles;

        ++Counters_->QueryError;

        Log_.Warning("%s: dr=%lu: got driver with error. deleting it", GetDsn(), driver->GetId());
    }

    void TDbPoolImpl::PutDriverWithTimeOut(TDbPoolImpl::TDriverPtr driver) {
        Y_ENSURE(driver, "got null driver: " << GetDsn());

        ++Counters_->QueryTimeout;

        Log_.Debug("%s: dr=%lu: got driver with timeout", GetDsn(), driver->GetId());
        PutDriverWithError(std::move(driver)); // TODO: should wait for results
    }

    void TDbPoolImpl::IncAllRequests() {
        ++Counters_->AllRequests;
    }

    void TDbPoolImpl::PutPingedDriver(TDbPoolImpl::TDriverPtr driver) {
        Y_ENSURE(driver, "got null driver: " << GetDsn());

        LastSuccessfulPing_ = TInstant::Now();
        PutGoodDriver(std::move(driver));
    }

    bool TDbPoolImpl::IsTimeToPing() const {
        return LastSuccessfulPing_ + Settings_.PingPeriod < TInstant::Now();
    }

    size_t TDbPoolImpl::NeedMoreDrivers() const {
        return Settings_.Size - Counters_->GoodHandles.GetValue();
    }

    size_t TDbPoolImpl::DriversCanBeCreated() const {
        return SizeLimit_ - Counters_->TotalHandles.GetCount();
    }

    void TDbPoolImpl::TryMakeStateUp() {
        State_.TryMake(NDbPool::TPoolState::Up);
    }

    void TDbPoolImpl::AddNewDriver(TDbPoolImpl::TDriverPtr driver) {
        Y_ENSURE(driver->IsReadyForQuery());

        ++Counters_->IdleHandles;
        ++Counters_->GoodHandles;
        Counters_->TotalHandles.Inc();

        Log_.Info("%s: dr=%lu: added new driver", GetDsn(), driver->GetId());
        IdleDrivers_.Enqueue(std::move(driver));
    }

    void TDbPoolImpl::Run() {
        struct TLogOnStart {
            TLogOnStart(const TDbPoolLog& log, const char* dsn) {
                log.Info("%s: routine thread is started", dsn);
            }
        } thread_local logOnStart(Log_, GetDsn());

        // 1. create new drivers
        // 2. ping drivers - only to keep them alive
        // 3. wait long queries - TODO

        TPoller poller;
        TDriverCreator creator(*this, poller);
        TDriverPinger pinger(*this, poller);

        bool continueLoop = false;
        auto touchProcessor = [&](auto& processor, const char* details) {
            try {
                continueLoop |= processor.DoesNeedMoreLoop();
            } catch (const std::exception& e) {
                continueLoop = true;
                Log_.Warning("%s: rotine exception in %s: %s", GetDsn(), details, e.what());
            }
        };

        do {
            continueLoop = false;

            try {
                poller.RunLoopOnce(TDuration::MilliSeconds(3));
            } catch (const std::exception& e) {
                continueLoop = true;
                Log_.Error("%s: poller error: %s", GetDsn(), e.what());
            }

            touchProcessor(creator, "DriverCreator");
            touchProcessor(pinger, "DriverPinger");
            // TODO: 3.
        } while (continueLoop && !Stopping_.load(std::memory_order_relaxed));
    }

    const char* TDbPoolImpl::GetDsn() {
        return Destination_.Serialized.c_str();
    }

    TDriverDestination TDbPoolImpl::CreateDriverDestination(const NDbPool::TDbPoolSettings& settings) {
        // Here we check driver type and settings
        TDriverPtr dummy = IDriver::CreateDriver(settings.Dsn->Driver);
        dummy->Init(settings);
        return dummy->GetDestination();
    }

    size_t TDbPoolImpl::CreateSizeLimit(const NDbPool::TDbPoolSettings& settings) {
        return 2 * settings.Size;
    }

    TDbPoolCtx::TTimeStat TDbPoolImpl::CreateTimeStats(const NDbPool::TDbPoolSettings& settings) {
        return std::make_shared<NUnistat::TTimeStat>(
            "dbpool.db2_" + settings.Dsn->DisplayName + ".responsetime",
            NUnistat::TTimeStat::CreateDefaultBounds());
    }
}
