#include "driver_creator.h"

#include "db_pool_impl.h"

#include <passport/infra/libs/cpp/dbpool2/misc/utils.h>
#include <passport/infra/libs/cpp/dbpool2/misc/driver/driver_preparation.h>

#include <passport/infra/libs/cpp/dbpool2/poller.h>

namespace NPassport::NDbPool2 {
    TDriverEmbryo::TDriverEmbryo(TDbPoolImpl& parent, TPoller& poller)
        : Parent_(&parent)
        , Poller_(&poller)
        , Log_(Parent_->GetLogger())
        , Driver_(IDriver::CreateDriver(Parent_->GetSettings().Dsn->Driver))
        , StartTime_(TInstant::Now())
    {
        Log_.Debug("%s: dr=%lu: DriverCreator: create new driver", Parent_->GetDsn(), Driver_->GetId());

        Driver_->Init(Parent_->GetSettings());
        FutureConnect_ = Driver_->StartConnecting();
        Poller_->Add(Driver_);
    }

    TDriverEmbryo::~TDriverEmbryo() {
        if (Driver_) {
            Poller_->Forget(Driver_);
        }
    }

    bool TDriverEmbryo::IsTimeToLeave() {
        try {
            return IsTimeToLeaveImpl();
        } catch (const std::exception& e) {
            Log_.Debug("%s: dr=%lu: DriverCreator: exception: %s",
                       Parent_->GetDsn(),
                       Driver_->GetId(),
                       e.what());
            State_ = Error;
        }

        return true;
    }

    bool TDriverEmbryo::IsTimeToLeaveImpl() {
        if (Connecting == State_) {
            if (FutureConnect_.HasValue()) {
                ThrowOnError(FutureConnect_.GetValue(), Log_, []() {});
                Log_.Debug("%s: dr=%lu: DriverCreator: driver connected", Parent_->GetDsn(), Driver_->GetId());

                FuturePrepare_ = Prepare(*Parent_->GetSettings().Dsn, *Driver_);
                StartTime_ = TInstant::Now();
                State_ = Preparing;
            } else if (StartTime_ + Parent_->GetSettings().ConnectionTimeout < TInstant::Now()) {
                Log_.Debug("%s: dr=%lu: DriverCreator: connecting timeout", Parent_->GetDsn(), Driver_->GetId());
                State_ = Error;
            } else {
                return false; // waiting for response
            }
        }

        if (Preparing == State_) {
            if (FuturePrepare_.HasValue()) {
                ThrowOnError(FuturePrepare_.GetValue(), Log_, []() {});
                Log_.Debug("%s: dr=%lu: DriverCreator: driver prepared", Parent_->GetDsn(), Driver_->GetId());

                Poller_->Forget(Driver_);
                Parent_->AddNewDriver(std::move(Driver_));
                State_ = Prepared;
            } else if (StartTime_ + Parent_->GetSettings().QueryTimeout < TInstant::Now()) {
                Log_.Debug("%s: dr=%lu: DriverCreator: preparing timeout", Parent_->GetDsn(), Driver_->GetId());
                State_ = Error;
            } else {
                return false; // waiting for response
            }
        }

        return true; // nothing to do here
    }

    TDriverCreator::TDriverCreator(TDbPoolImpl& parent, TPoller& poller)
        : Parent_(parent)
        , Poller_(poller)
    {
    }

    bool TDriverCreator::DoesNeedMoreLoop() {
        size_t need = Parent_.NeedMoreDrivers();
        if (0 == need && Drivers_.empty()) {
            return false;
        }

        // Drop ready drivers
        Drivers_.erase(std::remove_if(Drivers_.begin(), Drivers_.end(), [](TDriverEmbryo& d) {
                           return d.IsTimeToLeave();
                       }), Drivers_.end());

        if (0 == need) {
            Parent_.TryMakeStateUp();
        }

        const size_t canBeCreated = Parent_.DriversCanBeCreated();
        need = std::min(need, canBeCreated);

        const size_t driversInflightLimit = 16; // TODO: get from settings
        for (size_t idx = 0; idx < need && Drivers_.size() < driversInflightLimit; ++idx) {
            TDriverEmbryo driver(Parent_, Poller_);

            if (!driver.IsTimeToLeave()) {
                // need to wait response
                Drivers_.push_back(std::move(driver));
            }
        }

        return true;
    }
}
