#include "driver_pinger.h"

#include "db_pool_impl.h"

#include <passport/infra/libs/cpp/dbpool2/misc/utils.h>

#include <passport/infra/libs/cpp/dbpool2/poller.h>

namespace NPassport::NDbPool2 {
    TDriverGuard::TDriverGuard(TDbPoolImpl& parent)
        : Parent_(&parent)
        , Driver_(Parent_->GetDriver())
    {
    }

    TDriverGuard::~TDriverGuard() {
        if (Driver_) {
            Parent_->PutDriverWithError(std::move(Driver_));
        }
    }

    ui64 TDriverGuard::GetDriverId() {
        Y_ENSURE(Driver_);
        return Driver_->GetId();
    }

    std::shared_ptr<IDriver>& TDriverGuard::MutateDriver() {
        Y_ENSURE(Driver_);
        return Driver_;
    }

    TPingSubject::TPingSubject(TDbPoolImpl& parent, TPoller& poller)
        : TDriverGuard(parent)
        , Poller_(&poller)
        , Log_(parent.GetLogger())
        , StartTime_(TInstant::Now())
    {
        Future_ = this->MutateDriver()->StartPinging();
        Poller_->Add(this->MutateDriver());
    }

    bool TPingSubject::IsTimeToLeave() {
        try {
            if (!IsFinished_) {
                return IsFinished_ = IsTimeToLeaveImpl();
            }
        } catch (const std::exception& e) {
            Log_.Debug("%s: dr=%lu: DriverPinger: exception: %s",
                       this->Parent_->GetDsn(),
                       this->GetDriverId(),
                       e.what());
            IsFinished_ = true;
        }

        return true;
    }

    bool TPingSubject::IsTimeToLeaveImpl() {
        if (!Future_.HasValue()) {
            if (StartTime_ + this->Parent_->GetSettings().QueryTimeout < TInstant::Now()) {
                Log_.Debug("%s: dr=%lu: DriverPinger: driver timeouted", this->Parent_->GetDsn(), this->GetDriverId());
                this->Parent_->PutDriverWithTimeOut(std::move(this->MutateDriver()));
                return true;
            }

            return false;
        }

        ThrowOnError(Future_.GetValue(), Log_, []() {});
        Log_.Debug("%s: dr=%lu: DriverPinger: ping succeed", this->Parent_->GetDsn(), this->GetDriverId());

        Poller_->Forget(this->MutateDriver());
        this->Parent_->PutPingedDriver(std::move(this->MutateDriver()));
        return true;
    }

    TDriverPinger::TDriverPinger(TDbPoolImpl& parent, TPoller& poller)
        : Parent_(parent)
        , Poller_(poller)
    {
    }

    bool TDriverPinger::DoesNeedMoreLoop() {
        if (PingSubject_ && PingSubject_->IsTimeToLeave()) {
            PingSubject_.reset();
        }

        if (!Parent_.IsTimeToPing()) {
            return false;
        }

        if (!PingSubject_) {
            PingSubject_.emplace(Parent_, Poller_);
        }
        return true;
    }
}
