#include "sqlite_driver.h"

#include <passport/infra/libs/cpp/dbpool/exception.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NDbPool2 {
    static void UnixTimeFunc(sqlite3_context* context, int argc, sqlite3_value** argv) {
        if (argc > 1) {
            sqlite3_result_error(context, "bad arguments", -1);
            return;
        }
        if (0 == argc) {
            sqlite3_result_int(context, (int)time(nullptr));
            return;
        }
        const char* data = (const char*)sqlite3_value_text(argv[0]);
        struct tm tm {};
        if (nullptr == strptime(data, "%Y-%m-%d %H:%M:%S", &tm)) {
            sqlite3_result_error(context, "bad format", -1);
            return;
        }
        sqlite3_result_int(context, (int)mktime(&tm));
    }

    static bool IsBlob(const char* type) {
        return 0 == strcasecmp(type, "TINYBLOB") ||
               0 == strcasecmp(type, "MEDIUMBLOB") ||
               0 == strcasecmp(type, "LONGBLOB") ||
               0 == strcasecmp(type, "BLOB");
    }

    TSqliteDriver::TSqliteDriver()
        : Handle_(nullptr, sqlite3_close)
    {
    }

    TSqliteDriver::~TSqliteDriver() = default;

    void TSqliteDriver::OnPollEvent() {
        Y_FAIL("must be unreachable");
    }

    SOCKET TSqliteDriver::GetSocket() const {
        return DUMMY_SOCKET;
    }

    void TSqliteDriver::Init(const TDriverSettings& settings) {
        Settings_ = settings;

        Settings_.SerializedDestination = NUtils::CreateStr("db2;driver=sqlite;db=", Settings_.Dsn->Db, ";");
    }

    IDriver::TResultFuture<bool> TSqliteDriver::StartConnecting() {
        const TString n = NUtils::CreateStr("file:", Settings_.Dsn->Db, "?mode=ro&immutable=1");
        sqlite3* s = nullptr;
        if (SQLITE_OK != sqlite3_open_v2(n.c_str(), &s, SQLITE_OPEN_READONLY | SQLITE_OPEN_URI, nullptr)) {
            if (s) {
                sqlite3_close(s);
            }
            ythrow NDbPool::TException(Settings_.SerializedDestination) << "failed to connect to sqlite";
        }
        Handle_.reset(s);

        sqlite3_create_function(s, "UNIX_TIMESTAMP", -1, SQLITE_UTF8, nullptr, &UnixTimeFunc, nullptr, nullptr);

        return NThreading::MakeFuture<TErrorOr<bool>>(true);
    }

    bool TSqliteDriver::IsReadyForQuery() const noexcept {
        return (bool)Handle_;
    }

    TDriverDestination TSqliteDriver::GetDestination() const {
        return {Settings_.SerializedDestination, GetId()};
    }

    IDriver::TResultFuture<bool> TSqliteDriver::StartPinging() {
        return NThreading::MakeFuture<TErrorOr<bool>>(true);
    }

    TString TSqliteDriver::EscapeQueryParam(const TString& str) const {
        std::unique_ptr<char, std::function<void(char*)>> to(
            sqlite3_mprintf("%q", str.c_str()),
            sqlite3_free);
        return to ? TString(to.get()) : TString();
    }

    IDriver::TResultType TSqliteDriver::StartSendingQuery(NDbPool::TQuery&& query) {
        sqlite3_stmt* stmt;
        if (SQLITE_OK != sqlite3_prepare(Handle_.get(), query.Query().data(), query.Query().size(), &stmt, nullptr)) {
            return NThreading::MakeFuture<TErrorOr<NDbPool::TTable>>(GetErrorInfo());
        }

        int rc = sqlite3_step(stmt);
        int ncols = sqlite3_column_count(stmt);

        std::vector<bool> columnBlobs;
        if (SQLITE_ROW == rc) {
            for (int i = 0; i < ncols; ++i) {
                columnBlobs.push_back(IsBlob(sqlite3_column_decltype(stmt, i)));
            }
        }

        NDbPool::TTable result;
        while (SQLITE_ROW == rc) {
            NDbPool::TRow row;
            row.reserve(ncols);

            for (int idx = 0; idx < ncols; ++idx) {
                const char* res = (const char*)sqlite3_column_text(stmt, idx);
                row.emplace_back(TString(res ? res : ""),
                                 !res,
                                 columnBlobs.at(idx));
            }
            result.push_back(std::move(row));

            rc = sqlite3_step(stmt);
        }

        if (SQLITE_OK != sqlite3_finalize(stmt)) {
            return NThreading::MakeFuture<TErrorOr<NDbPool::TTable>>(GetErrorInfo());
        }

        return NThreading::MakeFuture<TErrorOr<NDbPool::TTable>>(std::move(result));
    }

    TDriverError TSqliteDriver::GetErrorInfo() {
        const char* res = sqlite3_errmsg(Handle_.get());
        return TDriverError{
            TDriverDestination{
                Settings_.SerializedDestination,
            },
            res ? res : "<no_sqlite_error>",
            sqlite3_errcode(Handle_.get()),
        };
    }
}
