#pragma once

#include <util/generic/string.h>
#include <util/generic/yexception.h>
#include <util/stream/str.h>

namespace NPassport::NDbPool2 {
    class TSqlQuery: public TString {
    public:
        template <typename T>
        explicit TSqlQuery(T&& query)
            : TString(std::forward<T>(query))
        {
        }
    };

    template <class Handle>
    class TSqlQueryBuilder {
    public:
        TSqlQueryBuilder(Handle* handle = nullptr, size_t reserve = 128)
            : handle_(handle)
        {
            query_.Reserve(reserve);
        }

        TSqlQueryBuilder& common(TStringBuf sql) {
            query_ << sql;
            return *this;
        }

        template <typename T>
        TSqlQueryBuilder& param(T value) {
            if constexpr (std::is_integral_v<T>) {
                static_assert(!std::is_same_v<T, ui8>, "ui8 serialized as char, not number");
                static_assert(!std::is_same_v<T, i8>, "i8 serialized as char, not number");
                query_ << value;
            } else if (TIsContainer<T>::value_) {
                for (const auto& p : value) {
                    param(p);
                    query_ << ",";
                }
                if (!value.empty()) {
                    query_.Str().pop_back();
                }
            } else {
                struct TDummy;
                static_assert(std::is_same_v<T, TDummy>, "Type T is not supported");
            }

            return *this;
        }

        TSqlQueryBuilder& param(const TString& string) {
            Y_ENSURE(handle_);
            query_ << "'" << handle_->escapeQueryParam(string) << "'";
            return *this;
        }

        TSqlQuery finish() {
            Y_ENSURE(query_);
            return TSqlQuery(std::move(query_.Str()));
        }

    private:
        template <class T>
        class TIsContainer {
            using TYesType = char[1];
            using TNoType = char[2];

            template <typename C>
            static TYesType& testBegin(decltype(&C::begin));
            template <typename C>
            static TNoType& testBegin(...);

            template <typename C>
            static TYesType& testEnd(decltype(&C::end));
            template <typename C>
            static TNoType& testEnd(...);

        public:
            static constexpr bool value_ = sizeof(testBegin<T>(0)) == sizeof(TYesType) &&
                                           sizeof(testEnd<T>(0)) == sizeof(TYesType);
        };

    private:
        TStringStream query_;
        Handle* handle_;
    };
}
