#include "fetcher.h"

#include <passport/infra/libs/cpp/auth_core/oauth_token_parser.h>
#include <passport/infra/libs/cpp/auth_core/sessionutils.h>
#include <passport/infra/libs/cpp/keyutils/keys.h>
#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <library/cpp/tvmauth/client/facade.h>

#include <unordered_map>

namespace NPassport::NGammaFetcher {
    TFetcher::TGammas TFetcher::FetchGammas(TIssPartFetcherPtr issPartsFetcher,
                                            const NXml::TConfig& config,
                                            const TString& xpath) {
        if (!config.AsBool(xpath + "/gamma_fetcher/enabled", false)) {
            if (config.AsBool(xpath + "/gamma_fetcher/use_keyrings", true)) {
                CheckKeyringsCacheBehaviour();
            }
            return {};
        }

        return TGammas{
            .Gammas = InitGammas(std::move(issPartsFetcher), config, xpath),
            .SigningGamma = config.AsNum<NAuth::TGammaId>(xpath + "/gamma_fetcher/items/default_id"),
        };
    }

    static const std::unordered_map<TString, NAuth::EEnitityType> TYPE_MAPPING = {
        {"session", NAuth::EEnitityType::Session},
        {"smth", NAuth::EEnitityType::Smth},
        {"oauth", NAuth::EEnitityType::OAuth},
        {"other", NAuth::EEnitityType::Other},
    };

    NAuth::EEnitityType TFetcher::GetType(const TString& type, const TString& path) {
        auto it = TYPE_MAPPING.find(type);

        Y_ENSURE(it != TYPE_MAPPING.end(),
                 "type '" << type << "' is unknown in " << path);

        return it->second;
    }

    static TString PrintIds(const TIssNkmsClient::TGammaParts& parts) {
        TString res;

        for (const auto& [key, value] : parts) {
            NUtils::AppendSeparated(res, ',', ToString(key));
        }

        return res;
    }

    NAuth::TGammaKeeperSettings::TGammas TFetcher::InitGammas(TIssPartFetcherPtr issPartsFetcher,
                                                              const NXml::TConfig& config,
                                                              const TString& xpath) {
        const TIssNkmsClient::TGammaParts issParts =
            FetchIssParts(std::move(issPartsFetcher), config, xpath);
        TLog::Debug() << "GammaFetcher: succeed to read ISS gammas";
        const TIssNkmsClient::TGammaParts passpParts =
            TIssNkmsClient::ParseResponse(NUtils::ReadFile(config.AsString(xpath + "/gamma_fetcher/passp_parts_path")));
        TLog::Debug() << "GammaFetcher: succeed to read PASSP gammas";

        NAuth::TGammaKeeperSettings::TGammas res;

        for (const TString& xitem : config.SubKeys(xpath + "/gamma_fetcher/items/item")) {
            NAuth::TGamma gamma =
                ReadSingleGamma(issParts, passpParts, config, xitem);

            const NAuth::TGammaId id = gamma.Id;
            auto [it, ok] = res.emplace(id, std::move(gamma));
            Y_ENSURE(ok, "there is duplicated gamma with id " << id);

            TLog::Debug() << "GammaFetcher: succeed: gamma id=" << id
                          << " with hash " << NUtils::Bin2hex(NUtils::TCrypto::Md5(*it->second.Body));
        }

        return res;
    }

    TIssNkmsClient::TGammaParts TFetcher::FetchIssParts(TIssPartFetcherPtr issPartsFetcher,
                                                        const NXml::TConfig& config,
                                                        const TString& xpath)
    {
        try {
            TIssNkmsClient::TGammaParts res = TIssNkmsClient::ParseResponse(issPartsFetcher->GetFromCache());

            for (const TString& xitem : config.SubKeys(xpath + "/gamma_fetcher/items/item")) {
                NAuth::TGammaId id = config.AsNum<NAuth::TGammaId>(xitem + "/@id");
                Y_ENSURE(res.contains(id),
                         "missing gamma id=" << id);
            }

            return res;
        } catch (const std::exception& e) {
            TLog::Warning() << "GammaFetcher: cache for iss parts cannot be used: " << e.what();
        }

        return TIssNkmsClient::ParseResponse(issPartsFetcher->Fetch());
    }

    NAuth::TGamma TFetcher::ReadSingleGamma(const TIssNkmsClient::TGammaParts& issParts,
                                            const TIssNkmsClient::TGammaParts& passpParts,
                                            const NXml::TConfig& config,
                                            const TString& xpath)
    {
        NAuth::TGamma res;

        const TString xtypes = xpath + "/types_to_check";
        for (const TString& type : NUtils::ToVector(config.AsString(xtypes, ""), ',')) {
            res.TypesToCheck.insert(GetType(type, xtypes));
        }

        const TInstant timeBound = TInstant::Seconds(
            config.AsNum<time_t>(xpath + "/valid_upper_bound", 0));
        if (timeBound) {
            res.ValidUpperBound = timeBound;
        }

        res.Id = config.AsNum<NAuth::TGammaId>(xpath + "/@id");

        auto getPart = [id = res.Id, &config, &xpath](const TIssNkmsClient::TGammaParts& parts,
                                                      const TString& name) -> TString {
            auto it = parts.find(id);
            Y_ENSURE(it != parts.end(),
                     "missing id=" << id << " in " << name << " keys; got: " << PrintIds(parts));

            const TString expected = config.AsString(xpath + "/sha256/" + name + "_part");
            const TString actual = NUtils::Bin2hex(NUtils::TCrypto::Sha256(it->second));
            Y_ENSURE(actual == expected,
                     "gamma id=" << id << " hash mismatch from " << name
                                 << ". expected: " << expected << ", actual: " << actual);

            return it->second;
        };

        res.Body = std::make_unique<NSecretString::TSecretString>(
            getPart(issParts, "iss") + getPart(passpParts, "passp"));

        return res;
    }

    static const TString SOME_GAMMAS_NAME = "SOME_GAMMAS_NAME";
    static const TString SOME_GAMMAS_VALUE = "SOME_GAMMAS_VALUE";

    void TFetcher::CheckKeyringsCacheBehaviour() {
        auto read = [&](ui16 checkId) {
            try {
                TString value = NKeyutils::TReader::Read(SOME_GAMMAS_NAME);
                if (SOME_GAMMAS_VALUE == value) {
                    TLog::Info() << "GammaFetcher: #" << checkId << " succeed: read";
                } else {
                    TLog::Warning() << "GammaFetcher: #" << checkId
                                    << ". value from keyrings is unexpected. "
                                    << "expected: " << SOME_GAMMAS_VALUE
                                    << ". actual: " << value << ".";
                }
            } catch (const std::exception& e) {
                TLog::Warning() << "GammaFetcher: #1. failed to read: " << e.what();
            }
        };

        read(1);

        try {
            NKeyutils::TWriter::Write(SOME_GAMMAS_NAME, SOME_GAMMAS_VALUE);
            TLog::Info() << "GammaFetcher: #2 succeed: write";
        } catch (const std::exception& e) {
            TLog::Warning() << "GammaFetcher: #2. failed to write: " << e.what();
        }

        read(3);
    }
}
