#include "iss_nkms_client.h"

#include "logger.h"

#include <passport/infra/libs/cpp/json/config.h>
#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/crypto/rsa.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <library/cpp/http/simple/http_client.h>

namespace NPassport::NGammaFetcher {
    TString TIssNkmsClient::TRequest::GetUrl() const {
        return NUtils::CreateStr(Host, ':', Port, Path);
    }

    TString TIssNkmsClient::Fetch(const TString& url,
                                  const TString& nameSpace,
                                  const TString& envType,
                                  const TString& serviceTicket,
                                  const TString& rsaKeysDir,
                                  const TTskvLog& log) {
        const TRequest request = BuildRequest(url, nameSpace, envType, serviceTicket);

        try {
            TIssNkmsClient::TEncryptedResponse encrypted = ParseRawResponse(request, FetchRaw(request));

            TString result = DecryptResponse(encrypted, rsaKeysDir);

            log.LogGammaFetch(true, request.GetUrl());

            return result;
        } catch (const std::exception& e) {
            log.LogGammaFetch(false, request.GetUrl(), e.what());
            throw;
        }
    }

    TIssNkmsClient::TGammaParts TIssNkmsClient::ParseResponse(const TStringBuf body) {
        const NJson::TConfig response = NJson::TConfig::ReadFromMemory(body);

        TGammaParts res;
        for (const TString& point : response.SubKeys("")) {
            const TString key = NJson::TConfig::GetKeyFromPath(point);

            NAuth::TGammaId id = 0;
            Y_ENSURE(TryIntFromString<10>(key, id),
                     "gamma id is invalid, got '" << key << "'");

            const TString decoded = NUtils::Base64ToBin(response.As<TString>(point));
            Y_ENSURE(decoded, "invalid base64 in value for id " << id);

            res.emplace(id, decoded);
        }

        return res;
    }

    TIssNkmsClient::TRequest TIssNkmsClient::BuildRequest(const TString& url,
                                                          const TString& nameSpace,
                                                          const TString& envType,
                                                          const TString& serviceTicket) {
        TSimpleHttpClientOptions opt(url);

        return TRequest{
            .Host = opt.Host(),
            .Port = opt.Port(),
            .Path = NUtils::CreateStr("/v1/secret/", nameSpace, "-", envType),
            .Headers = {
                {"X-Ya-Service-Ticket", serviceTicket},
            },
        };
    }

    TIssNkmsClient::TRawResponse TIssNkmsClient::FetchRaw(const TRequest& req) {
        TKeepAliveHttpClient client(req.Host, req.Port);

        TStringStream body;
        TRawResponse resp;
        resp.Code = client.DoGet(req.Path,
                                 &body,
                                 req.Headers,
                                 &resp.Headers);
        resp.Body = std::move(body.Str());

        return resp;
    }

    TIssNkmsClient::TEncryptedResponse TIssNkmsClient::ParseRawResponse(const TRequest& request,
                                                                        TRawResponse&& resp) {
        Y_ENSURE(resp.Code == 200,
                 "Failed to get keys: " << request.Host << ":" << request.Port << request.Path
                                        << ". code=" << resp.Code
                                        << ". " << resp.Body);

        auto getHeader = [&headers = resp.Headers](const TString& name) -> TString {
            const THttpInputHeader* header = headers.FindHeader(name);
            Y_ENSURE(header, "missing header '" << name << "'");
            return header->Value();
        };

        TEncryptedResponse res{
            .EncryptedSessKey = getHeader("X-Ya-Encrypted-Sess-Key"),
            .Iv = getHeader("X-Ya-IV"),
            .EncryptedBody = std::move(resp.Body),
        };

        const TString keyId = getHeader("X-Ya-Key-Id");
        Y_ENSURE(TryIntFromString<10>(keyId, res.KeyId),
                 "key id is not number: '" << keyId << "'");

        return res;
    }

    TString TIssNkmsClient::DecryptResponse(const TEncryptedResponse& response, const TString& rsaKeysDir) {
        const NUtils::TRsaPrivateEvp rsa =
            NUtils::TRsaPrivateEvp::FromPem(
                NUtils::ReadFile(
                    NUtils::CreateStr(rsaKeysDir, "/", response.KeyId, ".pem")));

        const TString decodedKey = NUtils::Base64ToBin(response.EncryptedSessKey);
        Y_ENSURE(decodedKey,
                 "session key is not valid base64: '" << response.EncryptedSessKey << "'");

        const TString sessKey = rsa.DecryptWithModes(
            decodedKey,
            {
                {"rsa_padding_mode", "oaep"},
                {"rsa_oaep_md", "sha256"},
                {"rsa_mgf1_md", "sha256"},
            });

        const TString iv = NUtils::Base64ToBin(response.Iv);
        Y_ENSURE(iv,
                 "iv is not valid base64: '" << response.Iv << "'");

        TString res;
        TString err;
        Y_ENSURE(NUtils::TCrypto::DecryptCbc(
                     sessKey,
                     iv,
                     response.EncryptedBody,
                     res,
                     &err),
                 "failed to decrypt: " << err);

        return res;
    }
}
