#include "iss_parts_fetcher.h"

#include "iss_nkms_client.h"
#include "logger.h"

#include <passport/infra/libs/cpp/keyutils/keys.h>
#include <passport/infra/libs/cpp/tvm/common/service_tickets.h>
#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/log/file_logger.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <library/cpp/tvmauth/client/facade.h>

namespace NPassport::NGammaFetcher {
    TIssPartBaseFetcher::TIssPartBaseFetcher(const NXml::TConfig& config, const TString& xpath)
        : Config_(config)
        , Xpath_(xpath)
    {
        GammaLogger_ = std::make_unique<TTskvLog>(
            Config_.CreateLogger(Xpath_ + "/gamma_fetcher/logger_gamma"));
    }

    TIssPartFetcherPtr TIssPartBaseFetcher::Create(const NXml::TConfig& config,
                                                   const TString& xpath,
                                                   std::shared_ptr<NTvmAuth::TTvmClient> tvmClient) {
        if (GetUrl(config, xpath).StartsWith("file://")) {
            return std::make_unique<TDiskFetcher>(config, xpath);
        }

        return std::make_unique<THttpFetcher>(config, xpath, std::move(tvmClient));
    }

    TString TIssPartBaseFetcher::GetFromCache() const {
        try {
            if (IsKeyringEnabled()) {
                TLog::Debug() << "GammaFetcher: Trying to read ISS gammas from keyring";
                TString res = NKeyutils::TReader::Read(GetKeyringsName());
                TLog::Debug() << "GammaFetcher: Succeed to read ISS gammas from keyring";
                return res;
            }
        } catch (const std::exception& e) {
            TLog::Warning() << "GammaFetcher: failed to read gammas from keyring: " << e.what();
        }

        return {};
    }

    TString TIssPartBaseFetcher::Fetch() const {
        const TString issParts = FetchImpl(GetUrl(Config_, Xpath_));
        TLog::Debug() << "GammaFetcher: Succeed to read ISS gammas";

        try {
            // Always update cache to force invalidation of previous cache version
            NKeyutils::TWriter::Write(GetKeyringsName(), issParts);
        } catch (const std::exception& e) {
            if (IsKeyringEnabled()) {
                throw;
            }
            TLog::Warning() << "GammaFetcher: failed to write gammas to keyring: " << e.what();
        }

        return issParts;
    }

    TString TIssPartBaseFetcher::GetUrl(const NXml::TConfig& config, const TString& xpath) {
        return config.AsString(xpath + "/gamma_fetcher/issnkms_url");
    }

    bool TIssPartBaseFetcher::IsKeyringEnabled() const {
        return Config_.AsBool(Xpath_ + "/gamma_fetcher/use_keyrings", true);
    }

    TString TIssPartBaseFetcher::GetKeyringsName() const {
        return Config_.AsString(Xpath_ + "/gamma_fetcher/keyrings_name", "BLACKBOX_GAMMAS_FOR_RANDOMS");
    }

    TDiskFetcher::TDiskFetcher(const NXml::TConfig& config, const TString& xpath)
        : TIssPartBaseFetcher(config, xpath)
    {
    }

    TString TDiskFetcher::FetchImpl(const TString& url) const {
        TStringBuf buf(url);
        Y_VERIFY(buf.SkipPrefix("file://"));
        TLog::Debug() << "GammaFetcher: Trying to read ISS gammas from disk";

        return NUtils::ReadFile(TString(buf));
    }

    THttpFetcher::THttpFetcher(const NXml::TConfig& config,
                               const TString& xpath,
                               std::shared_ptr<NTvmAuth::TTvmClient> tvmClient)
        : TIssPartBaseFetcher(config, xpath)
        , TvmClient_(std::move(tvmClient))
    {
        Y_ENSURE(TvmClient_, "tvm client must be configured for iss-nkms");
    }

    THttpFetcher::~THttpFetcher() = default;

    TString THttpFetcher::FetchImpl(const TString& url) const {
        TLog::Debug() << "GammaFetcher: Trying to read ISS gammas from ISS API";

        return TIssNkmsClient::Fetch(
            url,
            Config_.AsString(Xpath_ + "/gamma_fetcher/namespace", "blackbox_gamma"),
            Config_.AsString(Xpath_ + "/gamma_fetcher/env_type"),
            TvmClient_->GetServiceTicketFor(NTvmCommon::TServiceTickets::ISSNKMS_),
            Config_.AsString(Xpath_ + "/gamma_fetcher/iss_decrypt_keys_dir"),
            *GammaLogger_);
    }
}
