#include <passport/infra/libs/cpp/gamma_fetcher/fetcher.h>

#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/file.h>
#include <util/system/fs.h>

using namespace NPassport;
using namespace NPassport::NGammaFetcher;

Y_UNIT_TEST_SUITE(PasspUtFetcher) {
    class TTestIssPartFetcher: public IIssPartFetcher {
    public:
        TTestIssPartFetcher(const TString& value, const TString& cache = {})
            : Value(value)
            , Cache(cache)
        {
        }

        TString GetFromCache() const override {
            return Cache;
        }

        TString Fetch() const override {
            return Value;
        }

        TString Value;
        TString Cache;
    };

    Y_UNIT_TEST(getType) {
        UNIT_ASSERT_VALUES_EQUAL(NAuth::EEnitityType::Session,
                                 TFetcher::GetType("session", "some path"));
        UNIT_ASSERT_VALUES_EQUAL(NAuth::EEnitityType::OAuth,
                                 TFetcher::GetType("oauth", "some path"));
        UNIT_ASSERT_VALUES_EQUAL(NAuth::EEnitityType::Smth,
                                 TFetcher::GetType("smth", "some path"));
        UNIT_ASSERT_VALUES_EQUAL(NAuth::EEnitityType::Other,
                                 TFetcher::GetType("other", "some path"));

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TFetcher::GetType("foo", "some path"),
            yexception,
            "type 'foo' is unknown in some path");
    }

    Y_UNIT_TEST(initGammas) {
        const TString tmpDir = "./tmp/";
        NFs::Remove(tmpDir);
        NFs::MakeDirectory(tmpDir);
        {
            TFileOutput f(tmpDir + "passp.keys");
            f << R"({"47": "qvOnFo0zr9UPa/EhQRXE9ibj+Oxpz7OPfhUQwu21ozk="})";
        }

        const NXml::TConfig config = NXml::TConfig::ReadFromMemory(R"(
<doc>
    <gamma_fetcher>
        <passp_parts_path>./tmp/passp.keys</passp_parts_path>
        <items>
            <item id="47">
                <valid_upper_bound>0</valid_upper_bound>
                <types_to_check>session,oauth</types_to_check>
                <sha256>
                    <passp_part>9a49f1deba106d1f91e7c84eea948086344514df7b3b8596cce317c14cad4625</passp_part>
                    <iss_part>588226df643afd2bea982282bd9446bb9d2b12c250289826a26c37a055d12fac</iss_part>
                </sha256>
            </item>
        </items>
    </gamma_fetcher>
</doc>
)");

        UNIT_ASSERT_VALUES_EQUAL(
            NAuth::TGammaKeeperSettings::TGammas({
                {
                    47,
                    NAuth::TGamma({
                        .TypesToCheck = {NAuth::EEnitityType::Session, NAuth::EEnitityType::OAuth},
                        .Id = 47,
                        .Body = std::make_shared<NSecretString::TSecretString>(
                            NUtils::Hex2bin("c1475891810c86ece819e341e3641e686818cdc7e94467ba72dfd052309d48cdaaf3a7168d33afd50f6bf1214115c4f626e3f8ec69cfb38f7e1510c2edb5a339")),
                    }),
                },
            }),

            TFetcher::InitGammas(
                std::make_unique<TTestIssPartFetcher>(R"({"47": "wUdYkYEMhuzoGeNB42QeaGgYzcfpRGe6ct/QUjCdSM0="})"),
                config,
                "/doc"));

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TFetcher::InitGammas(
                std::make_unique<TTestIssPartFetcher>(R"({"47": "wUdYkYEMhuzoGeNB42QeaGgYzcfpRGe6ct/QUjCdSM0="})"),
                NXml::TConfig::ReadFromMemory(R"(
<doc>
    <gamma_fetcher>
        <passp_parts_path>./tmp/passp.keys</passp_parts_path>
        <items>
            <item id="47">
                <valid_upper_bound>0</valid_upper_bound>
                <types_to_check>session,oauth</types_to_check>
                <sha256>
                    <passp_part>9a49f1deba106d1f91e7c84eea948086344514df7b3b8596cce317c14cad4625</passp_part>
                    <iss_part>588226df643afd2bea982282bd9446bb9d2b12c250289826a26c37a055d12fac</iss_part>
                </sha256>
            </item>
            <item id="47">
                <valid_upper_bound>0</valid_upper_bound>
                <types_to_check>oauth</types_to_check>
                <sha256>
                    <passp_part>9a49f1deba106d1f91e7c84eea948086344514df7b3b8596cce317c14cad4625</passp_part>
                    <iss_part>588226df643afd2bea982282bd9446bb9d2b12c250289826a26c37a055d12fac</iss_part>
                </sha256>
            </item>
        </items>
    </gamma_fetcher>
</doc>)"),
                "/doc"),
            yexception,
            "there is duplicated gamma with id 47");
    }

    Y_UNIT_TEST(fetchIssParts) {
        const NXml::TConfig cfgWith47 = NXml::TConfig::ReadFromMemory(R"(
<doc>
    <gamma_fetcher>
        <items>
            <item id="47"/>
        </items>
    </gamma_fetcher>
</doc>)");

        UNIT_ASSERT_VALUES_EQUAL(
            TIssNkmsClient::TGammaParts({
                {47, NUtils::Base64ToBin("bbbb")},
            }),
            TFetcher::FetchIssParts(
                std::make_unique<TTestIssPartFetcher>(R"({"47": "bbbb"})", "invalid json"),
                cfgWith47,
                "/doc"));

        UNIT_ASSERT_VALUES_EQUAL(
            TIssNkmsClient::TGammaParts({
                {47, NUtils::Base64ToBin("aaaa")},
            }),
            TFetcher::FetchIssParts(
                std::make_unique<TTestIssPartFetcher>(R"({"47": "bbbb"})", R"({"47": "aaaa"})"),
                cfgWith47,
                "/doc"));

        UNIT_ASSERT_VALUES_EQUAL(
            TIssNkmsClient::TGammaParts({
                {47, NUtils::Base64ToBin("bbbb")},
                {48, NUtils::Base64ToBin("cccc")},
            }),
            TFetcher::FetchIssParts(
                std::make_unique<TTestIssPartFetcher>(R"({"47": "bbbb", "48": "cccc"})", R"({"47": "aaaa"})"),
                NXml::TConfig::ReadFromMemory(R"(
<doc>
    <gamma_fetcher>
        <items>
            <item id="47"/>
            <item id="48"/>
        </items>
    </gamma_fetcher>
</doc>)"),
                "/doc"));

    }

    Y_UNIT_TEST(readSingleGamma) {
        const NXml::TConfig config = NXml::TConfig::ReadFromMemory(R"(
<doc>
    <item id="47">
        <valid_upper_bound>0</valid_upper_bound>
        <types_to_check>session,oauth</types_to_check>
        <sha256>
            <passp_part>9a49f1deba106d1f91e7c84eea948086344514df7b3b8596cce317c14cad4625</passp_part>
            <iss_part>588226df643afd2bea982282bd9446bb9d2b12c250289826a26c37a055d12fac</iss_part>
        </sha256>
    </item>
</doc>
)");

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TFetcher::ReadSingleGamma({}, {}, config, "/doc/item"),
            yexception,
            "missing id=47 in iss keys");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TFetcher::ReadSingleGamma({{47, "foo"}}, {}, config, "/doc/item"),
            yexception,
            "gamma id=47 hash mismatch from iss. expected: 588226df643afd2bea982282bd9446bb9d2b12c250289826a26c37a055d12fac, actual: 2c26b46b68ffc68ff99b453c1d30413413422d706483bfa0f98a5e886266e7ae");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TFetcher::ReadSingleGamma(
                {{47, NUtils::Base64ToBin("wUdYkYEMhuzoGeNB42QeaGgYzcfpRGe6ct/QUjCdSM0=")}},
                {},
                config,
                "/doc/item"),
            yexception,
            "missing id=47 in passp keys");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TFetcher::ReadSingleGamma(
                {{47, NUtils::Base64ToBin("wUdYkYEMhuzoGeNB42QeaGgYzcfpRGe6ct/QUjCdSM0=")}},
                {{47, "bar"}},
                config,
                "/doc/item"),
            yexception,
            "gamma id=47 hash mismatch from passp. expected: 9a49f1deba106d1f91e7c84eea948086344514df7b3b8596cce317c14cad4625, actual: fcde2b2edba56bf408601fb721fe9b5c338d10ee429ea04fae5511b68fbf8fb9");

        UNIT_ASSERT_VALUES_EQUAL(
            NAuth::TGamma({
                .TypesToCheck = {NAuth::EEnitityType::Session, NAuth::EEnitityType::OAuth},
                .Id = 47,
                .Body = std::make_shared<NSecretString::TSecretString>(
                    NUtils::Hex2bin("c1475891810c86ece819e341e3641e686818cdc7e94467ba72dfd052309d48cdaaf3a7168d33afd50f6bf1214115c4f626e3f8ec69cfb38f7e1510c2edb5a339")),
            }),
            TFetcher::ReadSingleGamma(
                {{47, NUtils::Base64ToBin("wUdYkYEMhuzoGeNB42QeaGgYzcfpRGe6ct/QUjCdSM0=")}},
                {{47, NUtils::Base64ToBin("qvOnFo0zr9UPa/EhQRXE9ibj+Oxpz7OPfhUQwu21ozk=")}},
                config,
                "/doc/item"));

        // check valid_upper_bound
        UNIT_ASSERT_VALUES_EQUAL(
            NAuth::TGamma({
                .ValidUpperBound = TInstant::Seconds(100500),
                .TypesToCheck = {NAuth::EEnitityType::Session, NAuth::EEnitityType::OAuth},
                .Id = 47,
                .Body = std::make_shared<NSecretString::TSecretString>(
                    NUtils::Hex2bin("c1475891810c86ece819e341e3641e686818cdc7e94467ba72dfd052309d48cdaaf3a7168d33afd50f6bf1214115c4f626e3f8ec69cfb38f7e1510c2edb5a339")),
            }),
            TFetcher::ReadSingleGamma(
                {{47, NUtils::Base64ToBin("wUdYkYEMhuzoGeNB42QeaGgYzcfpRGe6ct/QUjCdSM0=")}},
                {{47, NUtils::Base64ToBin("qvOnFo0zr9UPa/EhQRXE9ibj+Oxpz7OPfhUQwu21ozk=")}},
                NXml::TConfig::ReadFromMemory(R"(
<doc>
    <item id="47">
        <valid_upper_bound>100500</valid_upper_bound>
        <types_to_check>session,oauth</types_to_check>
        <sha256>
            <passp_part>9a49f1deba106d1f91e7c84eea948086344514df7b3b8596cce317c14cad4625</passp_part>
            <iss_part>588226df643afd2bea982282bd9446bb9d2b12c250289826a26c37a055d12fac</iss_part>
        </sha256>
    </item>
</doc>)"),
                "/doc/item"));
    }
}

using namespace NAuth;

template <>
void Out<TGamma>(IOutputStream& out, const TGamma& value) {
    out << "validUpperBound: ";
    if (value.ValidUpperBound) {
        out << *value.ValidUpperBound << Endl;
    } else {
        out << "NULL" << Endl;
    }

    for (auto t : value.TypesToCheck) {
        out << "type to check: " << t << Endl;
    }

    out << "id: " << value.Id << Endl;
    out << "body: " << NUtils::Bin2hex(*value.Body) << Endl;
}
