#include <passport/infra/libs/cpp/gamma_fetcher/iss_nkms_client.h>

#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <library/cpp/http/simple/http_client.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/file.h>
#include <util/system/fs.h>

using namespace NPassport;
using namespace NPassport::NGammaFetcher;

Y_UNIT_TEST_SUITE(PasspUtIssNkmsClient) {
    static const TString RSA_PRIVATE_KEY =
        R"(-----BEGIN RSA PRIVATE KEY-----
MIIEpAIBAAKCAQEA1Jeo8gMXIzb4LkgXlWnk1nrgcxm+CngNzGBRNMcAnSyD25rC
lU56s2I2tXWU4m9WusGcqVo+2qgt6FDOudhdb10AgvdIVGN26kpLzL/C91O3CeWa
A6LA9d3g33kP7IK9X6xT+aLyrjPB0HkCUwpAb8PyWqFXNu+aL9au7F/4QtX9Azdn
YQ1QjZWTN+oW+aqQvZRVMkE1R9+MfdIwwTAZ5VKOmNyDeo2I1pTLwTWczuwJJstd
1adpIHI2BXVvjHcQBq+sf0ZEAVzNrLE9eI8m4vLBufQOTQzeQdl/icob6i9m8ifl
8KTPI7jlGZs6qCw+jWmJsAIvm95avqel0cEAuwIDAQABAoIBACyZiJsFJPSBUr26
6B+zcIvCiZX8H49oslRfHIo6+Zj/vsXyiMH0De/WOe4Fte9vNj3F4ozw3uDWiZME
aOo6RxfW5gWTxTvXBhqO7aoNzORATtUnsxDyMxMhskyPxFR09S++InvrwIflWdLX
oDI1o2EPNafn0zk2OR2yJ051r8qtYddwrLjGEH2pK8EO4Yk4vIPShepoG973kNc+
6jdovZunnVTS320O/dVFRZkdLwSEMpinvS6/ImwLsFr4aGXJFVSPy6uEGZNNfLVV
OfnZsiICKlniTxyE1z50H3Vfvl3QGdQ+iIJHqZBAEJx6R9vpJPt9sY/2NhYyP3Zj
g9LdGCECgYEA+2vjIWaquouDPDqPhsKkhyJQts0b90uZ9nKv6T0fsei/9iHp+Kyy
qxoroMBmZj5Bwgua/guzxMszK+poK6aLyrF0zVcr7J35xUsLdUVejB+i3bHElN9b
r3v+0lf3sK4wCD0TxsWvMyYAm6Cd2RpAbKDxJx+GXYUs3G7zg3e56YsCgYEA2HbB
BuyJQPpQR35zJhAkE2PVQ/+hnY7fuOJX0g7Mjmjz644zDeMOCN9gWEKT44vR7233
FbrdLwIIkeK7bO6WNvJW2kEAYk92eKkcTuku7dlL0zLxNe+VOHfZoq8XJf8fH3gO
PhkssFgIEwTcmrs9tfIBnCk5VFMgSTcWy48YS5ECgYAnUI+LwYog8X/sAxw+bmFb
4DnO2/Prj57bssNfLXevUj3yNwtnH1ow9z6rPGrGwiV2OmpaH2pU85Fk5Jq1eK2T
XHxvP5pWvb045Ks+A0LtlmTZVdrvjJ3BqxoFuku5DQPlxec5xd/Hl8GwlEUalchN
ND2BaKSSKykArzjU7fvcUQKBgQDJMBbYSKCJNB3JVU3Q4s+k9fUR7lZYCqMFQ9o7
8mVNN17+YHwzPAWfWX5Cih7KLvpAfwRXvgY12r38Wa+530beav8ue5vPw+to2kTl
UvW7O2uuMHrgln0qrroo6nOpbSWQKhWSwfM3M5rNkoc3iTAiG3xDmE0gguJPKlSV
fbonAQKBgQC0twv36cvY9JAJmjOWu1HspFI5ltfGX5gqIz/OniPQcIh0SyW0ubSi
J2UhZeefvESKlbkGcLj+OSIS3+j/jxR+WSKJfRHqQWUmF/1yZ/0zY/iMP1VucogB
1L3sQfs+Htbzrj9GOdNn6+AIByTfUqD5k166RJu+Xne5yM0xOL0KWw==
-----END RSA PRIVATE KEY-----
)";
    static const TString ENRYPTED_SESSION_KEY = "Xn3xAkNW4NhpHXSA/hHFj5WY+L1xscqaEMTgNIZ0hMsWkUrHv7TOLiHfRtnz0ISQS7iMp0Utsa2VRETT1qjEG6Ji0IfGgyh2TlC7GMIf7/t3ZD7W4kRxMND9YXqrObgBSY+2C5bQm4IoXHD92U17Er+DPGVXuGbZqi9g4kYyvjm5k8B96W/AchcITg3DMYjEbTg+ZAyi7AuDHo/KdpSO6/a9H93m/WPHiEWxrvZyXL0m3bBvH9TTAK7q4C9D9SZqiL+v1ePQJWTqHbK/V4Slzn0pGqIQchdgCVBe3BGCk3TL4eYlTfK0MdkzC0QO1CXP4Wf86g1rv0W9VHYRbObq9Q==";
    static const TString IV_ = "izNJDnmxA7nLYJGeoN0iPg==";

    static THttpHeaders CreateHeaders(THashMap<TString, TString> headers) {
        THttpHeaders res;
        for (const auto& [key, value] : headers) {
            res.AddHeader(key, value);
        }
        return res;
    }

    Y_UNIT_TEST(parseResponse) {
        struct TCase {
            TString CaseName;

            TString In;
            TString Except;
            TIssNkmsClient::TGammaParts Out;
        };

        std::vector<TCase> cases = {
            TCase{
                .CaseName = "invalid json",
                .In = "{",
                .Except = "Failed to parse json object",
            },
            TCase{
                .CaseName = "empty response",
                .In = "{}",
                .Out = {},
            },
            TCase{
                .CaseName = "invalid gamma id",
                .In = R"({"foo":"bar"})",
                .Except = "gamma id is invalid",
            },
            TCase{
                .CaseName = "too big gamma id",
                .In = R"({"66000":"bar"})",
                .Except = "gamma id is invalid",
            },
            TCase{
                .CaseName = "invalid gamma body",
                .In = R"({"123":[]})",
                .Except = "not string",
            },
            TCase{
                .CaseName = "empty gamma body",
                .In = R"({"123":""})",
                .Except = "value is empty",
            },
            TCase{
                .CaseName = "malformed gamma body",
                .In = R"({"123":"kek#1"})",
                .Except = "invalid base64 in value for id",
            },
            TCase{
                .CaseName = "good response",
                .In = R"({"123":"Zm9v", "64000":"YmFy"})",
                .Out = {
                    {123, "foo"},
                    {64000, "bar"},
                },
            },
        };

        for (TCase& c : cases) {
            if (c.Except) {
                UNIT_ASSERT_EXCEPTION_CONTAINS_C(
                    TIssNkmsClient::ParseResponse(c.In),
                    yexception,
                    c.Except,
                    c.CaseName);
                continue;
            }

            UNIT_ASSERT_VALUES_EQUAL_C(
                TIssNkmsClient::ParseResponse(c.In),
                c.Out,
                c.CaseName);
        }
    }

    Y_UNIT_TEST(buildRequest) {
        TIssNkmsClient::TRequest req =
            TIssNkmsClient::BuildRequest("http://some_url:467", "some_space", "some_env", "some ticket");

        UNIT_ASSERT_VALUES_EQUAL(req.Host, "http://some_url");
        UNIT_ASSERT_VALUES_EQUAL(req.Port, 467);
        UNIT_ASSERT_VALUES_EQUAL(req.Path, "/v1/secret/some_space-some_env");
        UNIT_ASSERT_VALUES_EQUAL(req.Headers,
                                 TKeepAliveHttpClient::THeaders({
                                     {"X-Ya-Service-Ticket", "some ticket"},
                                 }));

        req = TIssNkmsClient::BuildRequest("https://some_url", "some_space", "some_env", "some ticket");

        UNIT_ASSERT_VALUES_EQUAL(req.Host, "https://some_url");
        UNIT_ASSERT_VALUES_EQUAL(req.Port, 443);
        UNIT_ASSERT_VALUES_EQUAL(req.Path, "/v1/secret/some_space-some_env");
        UNIT_ASSERT_VALUES_EQUAL(req.Headers,
                                 TKeepAliveHttpClient::THeaders({
                                     {"X-Ya-Service-Ticket", "some ticket"},
                                 }));
    }

    Y_UNIT_TEST(parseRawResponse) {
        const TIssNkmsClient::TRequest req =
            TIssNkmsClient::BuildRequest("http://some_url:467", "some_space", "some_env", "some ticket");

        struct TCase {
            TString CaseName;

            TIssNkmsClient::TRawResponse In;
            TString Except;
            TIssNkmsClient::TEncryptedResponse Out;
        };
        std::vector<TCase> cases = {
            TCase{
                .CaseName = "bad http code",
                .In = {
                    .Code = 404,
                },
                .Except = "Failed to get keys: http://some_url:467/v1/secret/some_space-some_env. code=404.",
            },
            TCase{
                .CaseName = "missing X-Ya-Encrypted-Sess-Key",
                .In = {
                    .Code = 200,
                },
                .Except = "missing header 'X-Ya-Encrypted-Sess-Key'",
            },
            TCase{
                .CaseName = "missing X-Ya-IV",
                .In = {
                    .Code = 200,
                    .Headers = CreateHeaders({
                        {"X-Ya-Encrypted-Sess-Key", "kek#1"},
                    }),
                },
                .Except = "missing header 'X-Ya-IV'",
            },
            TCase{
                .CaseName = "missing X-Ya-Key-Id",
                .In = {
                    .Code = 200,
                    .Headers = CreateHeaders({
                        {"X-Ya-Encrypted-Sess-Key", "kek#1"},
                        {"X-Ya-IV", "kek#2"},
                    }),
                },
                .Except = "missing header 'X-Ya-Key-Id'",
            },
            TCase{
                .CaseName = "missing X-Ya-Key-Id",
                .In = {
                    .Code = 200,
                    .Headers = CreateHeaders({
                        {"X-Ya-Encrypted-Sess-Key", "kek#1"},
                        {"X-Ya-IV", "kek#2"},
                        {"X-Ya-Key-Id", "kek#3"},
                    }),
                },
                .Except = "key id is not number: 'kek#3'",
            },
            TCase{
                .CaseName = "missing X-Ya-Key-Id",
                .In = {
                    .Code = 200,
                    .Headers = CreateHeaders({
                        {"X-Ya-Encrypted-Sess-Key", "kek#1"},
                        {"X-Ya-IV", "kek#2"},
                        {"X-Ya-Key-Id", "4200100999"},
                    }),
                    .Body = "kek#4",
                },
                .Out = {
                    .KeyId = 4200100999,
                    .EncryptedSessKey = "kek#1",
                    .Iv = "kek#2",
                    .EncryptedBody = "kek#4",
                },
            },
        };

        for (TCase& c : cases) {
            if (c.Except) {
                UNIT_ASSERT_EXCEPTION_CONTAINS_C(
                    TIssNkmsClient::ParseRawResponse(req, std::move(c.In)),
                    yexception,
                    c.Except,
                    c.CaseName);
                continue;
            }

            TIssNkmsClient::TEncryptedResponse resp =
                TIssNkmsClient::ParseRawResponse(req, std::move(c.In));

            UNIT_ASSERT_VALUES_EQUAL_C(resp.KeyId, c.Out.KeyId, c.CaseName);
            UNIT_ASSERT_VALUES_EQUAL_C(resp.EncryptedSessKey, c.Out.EncryptedSessKey, c.CaseName);
            UNIT_ASSERT_VALUES_EQUAL_C(resp.Iv, c.Out.Iv, c.CaseName);
            UNIT_ASSERT_VALUES_EQUAL_C(resp.EncryptedBody, c.Out.EncryptedBody, c.CaseName);
        }
    }

    Y_UNIT_TEST(decryptResponse) {
        const TString tmpDir = "./tmp/";
        NFs::Remove(tmpDir);
        NFs::MakeDirectory(tmpDir);
        {
            TFileOutput f(tmpDir + "43.pem");
            f << RSA_PRIVATE_KEY;
        }

        struct TCase {
            TString CaseName;

            TIssNkmsClient::TEncryptedResponse In;
            TString Except;
            TString Out;
        };

        std::vector<TCase> cases = {
            TCase{
                .CaseName = "missing rsa key: 42",
                .In = {
                    .KeyId = 42,
                    .EncryptedSessKey = "kek#1",
                    .Iv = "kek#2",
                    .EncryptedBody = "kek#4",
                },
                .Except = tmpDir + "/42.pem",
            },
            TCase{
                .CaseName = "malformed session key",
                .In = {
                    .KeyId = 43,
                    .EncryptedSessKey = "kek#1",
                    .Iv = "kek#2",
                    .EncryptedBody = "kek#4",
                },
                .Except = "session key is not valid base64",
            },
            TCase{
                .CaseName = "rsa key mismatch",
                .In = {
                    .KeyId = 43,
                    .EncryptedSessKey = "aaaa",
                    .Iv = "kek#2",
                    .EncryptedBody = "kek#4",
                },
                .Except = "Failed to decrypt:",
            },
            TCase{
                .CaseName = "good session key, malformed iv",
                .In = {
                    .KeyId = 43,
                    .EncryptedSessKey = ENRYPTED_SESSION_KEY,
                    .Iv = "kek#2",
                    .EncryptedBody = "kek#4",
                },
                .Except = "iv is not valid base64",
            },
            TCase{
                .CaseName = "good session key and iv, session key mismatch",
                .In = {
                    .KeyId = 43,
                    .EncryptedSessKey = ENRYPTED_SESSION_KEY,
                    .Iv = IV_,
                    .EncryptedBody = "kek#4",
                },
                .Except = "failed to decrypt",
            },
            TCase{
                .CaseName = "success",
                .In = {
                    .KeyId = 43,
                    .EncryptedSessKey = ENRYPTED_SESSION_KEY,
                    .Iv = IV_,
                    .EncryptedBody = NUtils::Hex2bin("b90a148a8c494905a890d77249159a0dc9f39d30a68fe8ff7c7155a16372e468de52befef886e595ca0bd1976bbd74b3"),
                },
                .Out = R"({"1": "dGVzdGluZyAweERFQURCRUVG"})",
            },
        };

        for (TCase& c : cases) {
            if (c.Except) {
                UNIT_ASSERT_EXCEPTION_CONTAINS_C(
                    TIssNkmsClient::DecryptResponse(c.In, tmpDir),
                    yexception,
                    c.Except,
                    c.CaseName);
                continue;
            }

            UNIT_ASSERT_VALUES_EQUAL_C(
                TIssNkmsClient::DecryptResponse(c.In, tmpDir),
                c.Out,
                c.CaseName);
        }
    }
}

template <>
void Out<TIssNkmsClient::TGammaParts>(IOutputStream& out, const TIssNkmsClient::TGammaParts& value) {
    for (const auto& [key, value] : value) {
        out << key << "->" << NUtils::BinToBase64(value) << Endl;
    }
}
