#include "batch.h"

#include <passport/infra/libs/cpp/unistat/time_stat.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

#include <util/generic/yexception.h>
#include <util/stream/format.h>

namespace NPassport::NHbase {
    using TGeneratedMutation = apache::hadoop::hbase::thrift::Mutation;

    TBatch::TBatch(IHBasePool& pool,
                   const TSettings& settings,
                   TTimeStatsPtr stats,
                   const TString& logableCookies)
        : Pool_(pool)
        , Settings_(settings)
        , LogableCookies_(logableCookies)
        , Stats_(std::move(stats))
        , Backoff_(settings.MinBackoff, settings.MaxBackoff, 2, 0.1)
    {
    }

    TBatch::~TBatch() {
        try {
            Backoff_.Interrupt();
        } catch (...) {
        }
    }

    void TBatch::Add(const std::string& tableName, const TQueryArray& queries) {
        SplitAndSend(tableName, PrepareRows(queries));
    }

    TBatch::TPreparedRows TBatch::PrepareRows(const TQueryArray& queries) {
        TPreparedRows res;
        res.reserve(queries.size() / 2);

        for (const NHbase::TQuery& query : queries) {
            Y_ENSURE(query.Type == NHbase::TQuery::EType::Put);

            TGeneratedBatch& b = res.try_emplace(query.Row, TGeneratedBatch{}).first->second;

            b.Row = query.Row;

            b.Mutations.reserve(b.Mutations.size() + query.Params.size());
            for (const auto& [column, value] : query.Params) {
                TGeneratedMutation m;
                m.IsDelete = false;
                m.Column = column;
                m.Value = value;

                b.Mutations.push_back(std::move(m));
            }
        }

        return res;
    }

    void TBatch::SplitAndSend(const std::string& tableName, TBatch::TPreparedRows&& rows) {
        size_t currentMutationSize = 0;
        std::vector<TGeneratedBatch> batches;
        batches.reserve(std::min(Settings_.Limit, rows.size()));

        BatchCount_ += rows.size();

        for (auto& [_, batch] : rows) {
            currentMutationSize += batch.Mutations.size();
            MutationCount_ += batch.Mutations.size();

            batches.push_back(std::move(batch));

            if (currentMutationSize >= Settings_.Limit) {
                size_t prevSize = batches.size();
                Send(tableName, std::move(batches));

                batches.clear();
                batches.reserve(prevSize);

                currentMutationSize = 0;
            }
        }

        Send(tableName, std::move(batches));
    }

    void TBatch::Send(const std::string& tableName, std::vector<TGeneratedBatch>&& batches) {
        if (batches.empty()) {
            return;
        }

        TRequest r{
            .TableName = tableName,
            .Batches = std::move(batches),
            .Tries = Settings_.Retries,
        };

        while (r.Tries-- > 0) {
            if (SendBlocking(r)) {
                return;
            }

            Backoff_.Sleep();
            Backoff_.Increase();
        }

        ythrow yexception() << "Failed to mutate rows after all attempts (send)";
    }

    TClientGuard TBatch::GetClient() {
        TClientGuard cl = Pool_.GetClient();
        if (cl) {
            return cl;
        }

        const TInstant start = TInstant::Now();
        while (!cl && TInstant::Now() - start < Settings_.GetTimeout) {
            Sleep(TDuration::MilliSeconds(250));
            cl = Pool_.GetClient();
        }

        return {};
    }

    bool TBatch::SendBlocking(TBatch::TRequest& r) {
        ++RequestCount_;

        try {
            r.Client = GetClient();
            Y_ENSURE(r.Client, "Failed to get alive client for hbase");

            r.Client->MutateRows(r.TableName, r.Batches);
            ProcessResponse(r);

            return true;
        } catch (const std::exception& e) {
            TLog::Warning() << "Batch#" << Settings_.LogableIdx
                            << ": got exception on mutating rows (send): " << e.what()
                            << ". tries left:" << r.Tries
                            << ". cookies " << LogableCookies_;
        }

        return false;
    }

    void TBatch::ProcessResponse(TBatch::TRequest& r) {
        Backoff_.Decrease();

        size_t mutations = 0;
        for (const TGeneratedBatch& b : r.Batches) {
            mutations += b.Mutations.size();
        }

        SucceedCount_ += r.Batches.size();

        TDuration took = TInstant::Now() - r.Start;
        if (Settings_.VerboseLogs) {
            TLog::Debug() << "Batch#" << Settings_.LogableIdx
                          << ": succeed to mutate rows: " << mutations
                          << " mutations. Batch(es): "
                          << SucceedCount_ << "/" << BatchCount_
                          << " (+" << r.Batches.size() << ")"
                          << ". took " << HumanReadable(took)
                          << ". cookies " << LogableCookies_;
        }

        if (Stats_) {
            Stats_->Insert(took);
        }
    }
}
