#include "hbase_client.h"

#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NHbase {
    class TErrHandler {
    public:
        TErrHandler(std::unique_ptr<THBaseSession>* session)
            : Session_(session)
        {
        }

        ~TErrHandler() {
            if (Session_) {
                Session_->reset();
            }
        }

        void Cleanup() {
            Session_ = nullptr;
        }

    private:
        std::unique_ptr<THBaseSession>* Session_;
    };

    THBaseSession::THBaseSession(const TClientSettings& settings)
        : Socket_(std::make_shared<TGeneratedSocket>(settings.Host, settings.Port))
        , Transport_(std::make_unique<apache::thrift::transport::TFramedTransport>(Socket_))
        , Protocol_(std::make_shared<apache::thrift::protocol::TBinaryProtocol>(Transport_))
    {
        Y_VERIFY(settings.ConnectTimeout, "Connect timeout must not be empty");
        Y_VERIFY(settings.SendTimeout, "Send timeout must not be empty");
        Y_VERIFY(settings.RecvTimeout, "Recv timeout must not be empty");

        Socket_->setConnTimeout(settings.ConnectTimeout.MilliSeconds());
        Socket_->setSendTimeout(settings.SendTimeout.MilliSeconds());
        Socket_->setRecvTimeout(settings.RecvTimeout.MilliSeconds());

        Client = std::make_unique<TGeneratedClient>(Protocol_);

        Transport_->open();
    }

    THBaseClient::THBaseClient(const TClientSettings& settings)
        : Settings_(settings)
    {
    }

    bool THBaseClient::IsOpen() const {
        return bool(Session_);
    }

    bool THBaseClient::TryOpen() {
        try {
            return Open();
        } catch (const std::exception& e) {
            TLog::Warning() << "Failed to reopen hbase connection: "
                            << Settings_.Host << ". "
                            << e.what();
        }

        return false;
    }

    void THBaseClient::MutateRows(const std::string& tableName, const std::vector<TGeneratedBatch>& rowBatches) {
        Y_ENSURE(Session_);

        HandleErr([&]() {
            Session_->Client->MutateRows(tableName, rowBatches, {});
        });
    }

    bool THBaseClient::Open() {
        const TInstant now = TInstant::Now();
        if (Session_ && now < Session_->LastAction + Settings_.ReopenAfterIdle) {
            return true;
        }
        Session_.reset();

        if (now < LastOpenAttempt_ + Settings_.MinReopenPeriod) {
            return false;
        }

        LastOpenAttempt_ = now;
        Session_ = std::make_unique<THBaseSession>(Settings_);
        return true;
    }

    template <typename Func>
    void THBaseClient::HandleErr(Func func) {
        TErrHandler err(&Session_);

        func();

        err.Cleanup();

        if (Session_) {
            Session_->LastAction = TInstant::Now();
        }
    }
}
