#pragma once

#include <passport/infra/libs/cpp/hbase/generated/Hbase.h>

#include <thrift/protocol/TBinaryProtocol.h>
#include <thrift/transport/TBufferTransports.h>
#include <thrift/transport/TSocket.h>

#include <util/datetime/base.h>
#include <util/generic/string.h>

namespace NPassport::NHbase {
    using TGeneratedBatch = apache::hadoop::hbase::thrift::BatchMutation;
    using TGeneratedClient = apache::hadoop::hbase::thrift::HbaseClient;
    using TGeneratedSocket = apache::thrift::transport::TSocket;
    using TGeneratedTransport = apache::thrift::transport::TTransport;
    using TGeneratedProtocol = apache::thrift::protocol::TProtocol;

    struct TClientSettings {
        const TString Host;
        ui16 Port;
        TDuration MinReopenPeriod;
        TDuration ConnectTimeout;
        TDuration SendTimeout;
        TDuration RecvTimeout;
        TDuration ReopenAfterIdle;
    };

    class THBaseSession {
    public:
        THBaseSession(const TClientSettings& settings);

        std::unique_ptr<TGeneratedClient> Client;
        TInstant LastAction;

    private:
        std::shared_ptr<TGeneratedSocket> Socket_;
        std::shared_ptr<TGeneratedTransport> Transport_;
        std::shared_ptr<TGeneratedProtocol> Protocol_;
    };

    class IHBaseClient {
    public:
        virtual ~IHBaseClient() = default;
        virtual void SetBusy(bool val) = 0;
        virtual void MutateRows(const std::string& tableName,
                                const std::vector<TGeneratedBatch>& rowBatches) = 0;
    };

    class THBaseClient: public IHBaseClient, TNonCopyable {
    public:
        THBaseClient(const TClientSettings& settings);
        ~THBaseClient() = default;

        bool IsOpen() const;
        bool TryOpen();

        void SetBusy(bool val) override {
            IsBusy_ = val;
        }

        bool IsBusy() const {
            return IsBusy_;
        }

    public: // proxy some functions
        void MutateRows(const std::string& tableName, const std::vector<TGeneratedBatch>& rowBatches) override;

    private:
        bool Open();

        template <typename Func>
        void HandleErr(Func func);

    private:
        const TClientSettings Settings_;

        std::unique_ptr<THBaseSession> Session_;

        TInstant LastOpenAttempt_;

        bool IsBusy_ = false;
    };

    class TClientGuard: TMoveOnly {
    public:
        TClientGuard() = default;

        TClientGuard(IHBaseClient* client)
            : Client_(client)
        {
            if (Client_) {
                Client_->SetBusy(true);
            }
        }

        TClientGuard(TClientGuard&& o)
            : Client_(o.Client_)
        {
            o.Client_ = nullptr;
        }

        void operator=(TClientGuard&& o) {
            Close();
            Client_ = o.Client_;
            o.Client_ = nullptr;
        }

        ~TClientGuard() {
            Close();
        }

        operator bool() const noexcept {
            return Client_;
        }

        IHBaseClient& operator*() {
            Y_ENSURE(Client_, "Guard is already closed");
            return *Client_;
        }

        IHBaseClient* operator->() {
            Y_ENSURE(Client_, "Guard is already closed");
            return Client_;
        }

    private:
        void Close() {
            if (Client_) {
                Client_->SetBusy(false);
                Client_ = nullptr;
            }
        }

    private:
        IHBaseClient* Client_ = nullptr;
    };
}
