#include "hbase_pool.h"

#include <passport/infra/libs/cpp/utils/log/global.h>

#include <library/cpp/containers/stack_vector/stack_vec.h>

#include <thrift/TOutput.h>

#include <util/generic/yexception.h>
#include <util/random/easy.h>

namespace NPassport::NHbase::NPrivate {
    class TThriftOutputIniter {
    public:
        TThriftOutputIniter() {
            apache::thrift::GlobalOutput.setOutputFunction(Print);
        }

        static void Print(const char* msg) {
            TLog::Warning() << "Thrift: " << msg;
        }
    };
}

namespace NPassport::NHbase {
    THBasePool::THBasePool(const TSettings& settings) {
        TLog::Debug() << "HBase pool: creating";
        static NPrivate::TThriftOutputIniter output;

        Instances_.reserve(settings.Hosts.size());

        size_t good = 0;

        for (const TSettings::THost& h : settings.Hosts) {
            Instances_.push_back(std::make_unique<THBaseClient>(
                TClientSettings{
                    .Host = h.Host,
                    .Port = h.Port,
                    .MinReopenPeriod = settings.MinReopenPeriod,
                    .ConnectTimeout = settings.ConnectTimeout,
                    .SendTimeout = settings.SendTimeout,
                    .RecvTimeout = settings.RecvTimeout,
                    .ReopenAfterIdle = settings.ReopenAfterIdle,
                }));

            if (Instances_.back()->TryOpen()) {
                ++good;
            }
        }

        Y_ENSURE(good > 0, "Failed to create any connection to HBase");
        TLog::Info() << "HBase pool is up with " << good << " backends";
    }

    TClientGuard THBasePool::GetClient() {
        TStackVec<int, 32> indexes(Instances_.size());
        std::iota(indexes.begin(), indexes.end(), 0);

        while (!indexes.empty()) {
            size_t rnd = RandomNumber(indexes.size());

            THBaseClient* c = TryGet(indexes[rnd]);
            if (c) {
                return c;
            }

            indexes.erase(indexes.begin() + rnd);
        }

        return nullptr;
    }

    THBaseClient* THBasePool::TryGet(size_t idx) {
        THBaseClient* c = Instances_[idx].get();

        if (c->IsBusy()) {
            return nullptr;
        }

        c->TryOpen();
        return c->IsOpen() ? c : nullptr;
    }
}
