#include <passport/infra/libs/cpp/hbase/batch.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NHbase;

Y_UNIT_TEST_SUITE(Batch) {
    class TTestClient: public IHBaseClient {
    public:
        void SetBusy(bool) override {
        }

        void MutateRows(const std::string& tableName,
                        const std::vector<TGeneratedBatch>& rowBatches) override {
            Data.push_back(TData{.TableName = tableName, .Batches = rowBatches});
        }

        void Sort() {
            std::sort(Data.begin(), Data.end(), [](const auto& l, const auto& r) {
                return l.TableName < r.TableName;
            });

            for (TData& d : Data) {
                std::sort(d.Batches.begin(), d.Batches.end(), [](const auto& l, const auto& r) {
                    return l.Row < r.Row;
                });
            }
        }

        struct TData {
            std::string TableName;
            std::vector<TGeneratedBatch> Batches;
        };

        std::vector<TData> Data;
    };

    class TTestPool: public IHBasePool {
    public:
        TClientGuard GetClient() override {
            return &Client;
        }

        TTestClient Client;
    };

    struct TMutations {
        std::string Row;
        std::vector<std::pair<std::string, std::string>> Pairs;
    };

    template <typename T>
    void Check(const TMutations& m, const TGeneratedBatch& b, const T& loc) {
        UNIT_ASSERT_VALUES_EQUAL_C(m.Row, b.Row, loc);
        UNIT_ASSERT_VALUES_EQUAL_C(m.Pairs.size(), b.Mutations.size(), loc);

        for (size_t idx = 0; idx < m.Pairs.size(); ++idx) {
            UNIT_ASSERT_VALUES_EQUAL_C(m.Pairs.at(idx).first, b.Mutations.at(idx).Column, loc);
            UNIT_ASSERT_VALUES_EQUAL_C(m.Pairs.at(idx).second, b.Mutations.at(idx).Value, loc);
        }
    }

    Y_UNIT_TEST(common) {
        TTestPool pool;
        TBatch batch(pool, TBatch::TSettings{});

        TQueryArray q;
        q.push_back(TQuery{
            .Row = "row_1",
            .Params = {
                {"key1", "val1"},
                {"key2", "val2"},
            },
        });
        q.push_back(TQuery{
            .Row = "row_1",
            .Params = {
                {"key3", "val3"},
            },
        });
        batch.Add("table_name_1", q);

        pool.Client.Sort();
        const std::vector<TTestClient::TData>& d = pool.Client.Data;
        UNIT_ASSERT_VALUES_EQUAL(1, d.size());
        UNIT_ASSERT_VALUES_EQUAL("table_name_1", d.at(0).TableName);
        UNIT_ASSERT_VALUES_EQUAL(1, d.at(0).Batches.size());
        Check(TMutations{
                  .Row = "row_1",
                  .Pairs = {
                      {"key1", "val1"},
                      {"key2", "val2"},
                      {"key3", "val3"},
                  },
              }, d.at(0).Batches.at(0), __LOCATION__);
        pool.Client.Data.clear();

        q.clear();
        q.push_back(TQuery{
            .Row = "row_2",
            .Params = {
                {"key4", "val4"},
            },
        });
        q.push_back(TQuery{
            .Row = "row_3",
            .Params = {
                {"key5", "val5"},
            },
        });

        batch.Add("table_name_1", q);
        pool.Client.Sort();
        UNIT_ASSERT_VALUES_EQUAL(1, d.size());
        UNIT_ASSERT_VALUES_EQUAL("table_name_1", d.at(0).TableName);
        UNIT_ASSERT_VALUES_EQUAL(2, d.at(0).Batches.size());
        Check(TMutations{
                  .Row = "row_2",
                  .Pairs = {
                      {"key4", "val4"},
                  },
              }, d.at(0).Batches.at(0), __LOCATION__);
        Check(TMutations{
                  .Row = "row_3",
                  .Pairs = {
                      {"key5", "val5"},
                  },
              }, d.at(0).Batches.at(1), __LOCATION__);
        pool.Client.Data.clear();
    }

    Y_UNIT_TEST(limit) {
        TTestPool pool;
        TBatch batch(pool, TBatch::TSettings{.Limit = 2});

        TQueryArray q;
        q.push_back(TQuery{
            .Row = "row_1",
            .Params = {
                {"key1", "val1"},
                {"key2", "val2"},
            },
        });
        q.push_back(TQuery{
            .Row = "row_2",
            .Params = {
                {"key4", "val4"},
            },
        });
        q.push_back(TQuery{
            .Row = "row_3",
            .Params = {
                {"key5", "val5"},
            },
        });
        q.push_back(TQuery{
            .Row = "row_1",
            .Params = {
                {"key3", "val3"},
            },
        });
        batch.Add("table_name_1", q);

        UNIT_ASSERT_VALUES_EQUAL(2, pool.Client.Data.size());
        // order of mutations is not determined because of unordered_map - so we cannot check grouping
    }
}
