#include "config.h"

namespace NPassport::NDaemon {
    TConfig::TConfig(const NXml::TConfig& config) {
        const TString xpath = "/config";

        Daemon_ = ParseDaemon(config, xpath);
        HttpCommon_ = ParseHttpCommon(config, xpath);
        HttpUnistat_ = ParseHttpUnistat(config, xpath);
    }

    TConfig::TConfig(const NJson::TConfig& config) {
        Daemon_ = ParseDaemon(config, "");
        HttpCommon_ = ParseHttpCommon(config, "");
        HttpUnistat_ = ParseHttpUnistat(config, "");
    }

    TConfig::TConfig(TConfig&&) noexcept = default;
    TConfig::~TConfig() = default;

    TConfig::TDaemon TConfig::ParseDaemon(const NXml::TConfig& config, const TString& xpath) {
        TConfig::TDaemon res;

        if (config.Contains(xpath + "/daemon/monitor_port")) {
            res.MonitorPort = config.AsNum<ui16>(xpath + "/daemon/monitor_port");
        }
        if (config.Contains(xpath + "/daemon/pidfile")) {
            res.PidPath = config.AsString(xpath + "/daemon/pidfile");
        }
        res.EnableAllocatorDefrag = config.AsBool(xpath + "/daemon/enable_allocator_defragmentation", true);

        return res;
    }

    static std::vector<TConfig::THttpCommon::TInterface> ParseInterfaces(const NXml::TConfig& config, const TString& xpath) {
        std::vector<TConfig::THttpCommon::TInterface> res;

        for (const TString& path : config.SubKeys(xpath + "/http_daemon/port")) {
            size_t reuseCount = config.AsNum<size_t>(path + "/@reuse_count", 1);
            Y_ENSURE(reuseCount > 0);
            res.push_back({config.AsNum<ui16>(path), reuseCount});
        }

        Y_ENSURE(!res.empty(), "At least one http port for listening is required");

        return res;
    }

    TConfig::THttpCommon TConfig::ParseHttpCommon(const NXml::TConfig& config, const TString& xpath) {
        THttpCommon res;

        res.Interfaces = ParseInterfaces(config, xpath);

        res.ListenAddress = config.AsString(xpath + "/http_daemon/listen_address", "localhost");
        res.Threads = config.AsInt(xpath + "/http_daemon/threads", 4);
        res.MaxConnections = config.AsInt(xpath + "/http_daemon/max_connections", 4096);
        res.MaxQueueSize = config.AsInt(xpath + "/http_daemon/max_queue_size", 4096);
        res.MaxDelay = TDuration::MilliSeconds(
            config.AsInt(xpath + "/http_daemon/max_delay", 1000));
        res.KeepAlive = config.AsBool(xpath + "/http_daemon/keep_alive", true);

        res.ResponseTimeHeaders = ParseResponseTimeHeaders(config, xpath + "/http_daemon/response_time_headers");

        return res;
    }

    std::optional<TConfig::THttpUnistat> TConfig::ParseHttpUnistat(const NXml::TConfig& config, const TString& xpath) {
        if (!config.Contains(xpath + "/unistat")) {
            return {};
        }

        THttpUnistat res;

        res.ListenAddress = config.AsString(xpath + "/unistat/listen_address", "localhost");
        res.Port = config.AsNum<ui16>(xpath + "/unistat/port");
        res.Threads = config.AsInt(xpath + "/unistat/threads", 2);
        res.MaxConnections = config.AsInt(xpath + "/unistat/max_connections", 10);
        res.MaxQueueSize = config.AsInt(xpath + "/unistat/max_queue_size", 10);
        res.MaxDelay = TDuration::MilliSeconds(config.AsInt(xpath + "/unistat/max_delay", 1000));
        res.KeepAlive = config.AsBool(xpath + "/unistat/keep_alive", false);
        res.Path = config.AsString(xpath + "/unistat/path");

        return res;
    }

    TConfig::TResponseTimeHeaders TConfig::ParseResponseTimeHeaders(const NXml::TConfig& config, const TString& xpath) {
        TResponseTimeHeaders res;

        res.InQueue = config.AsString(xpath + "/in_queue", "");
        res.InHandler = config.AsString(xpath + "/in_handler", "");

        return res;
    }

    TConfig::TDaemon TConfig::ParseDaemon(const NJson::TConfig& config, const TString& jsonPoint) {
        TConfig::TDaemon res;

        if (config.Contains(jsonPoint + "/daemon/monitor_port")) {
            res.MonitorPort = config.As<ui32>(jsonPoint + "/daemon/monitor_port");
        }
        if (config.Contains(jsonPoint + "/daemon/pidfile")) {
            res.PidPath = config.As<TString>(jsonPoint + "/daemon/pidfile");
        }
        res.EnableAllocatorDefrag = config.As<bool>(jsonPoint + "/daemon/enable_allocator_defragmentation", true);

        return res;
    }

    static std::vector<TConfig::THttpCommon::TInterface> ParseInterfaces(const NJson::TConfig& config, const TString& jsonPoint) {
        std::vector<TConfig::THttpCommon::TInterface> res;

        for (const TString& path : config.SubKeys(jsonPoint + "/http_daemon/ports")) {
            size_t reuseCount = config.As<size_t>(path + "/reuse_count", 1);
            Y_ENSURE(reuseCount > 0);
            res.push_back({(ui16)config.As<ui32>(path + "/port"), reuseCount});
        }

        Y_ENSURE(!res.empty(), "At least one http port for listening is required");

        return res;
    }

    TConfig::THttpCommon TConfig::ParseHttpCommon(const NJson::TConfig& config, const TString& jsonPoint) {
        THttpCommon res;

        res.Interfaces = ParseInterfaces(config, jsonPoint);

        res.ListenAddress = config.As<TString>(jsonPoint + "/http_daemon/listen_address", "localhost");
        res.Threads = config.As<size_t>(jsonPoint + "/http_daemon/threads", 4);
        res.MaxConnections = config.As<size_t>(jsonPoint + "/http_daemon/max_connections", 4096);
        res.MaxQueueSize = config.As<size_t>(jsonPoint + "/http_daemon/max_queue_size", 4096);
        res.MaxDelay = TDuration::MilliSeconds(config.As<size_t>(jsonPoint + "/http_daemon/max_delay", 1000));
        res.KeepAlive = config.As<bool>(jsonPoint + "/http_daemon/keep_alive", true);

        res.ResponseTimeHeaders = ParseResponseTimeHeaders(config, jsonPoint + "/response_time_headers");

        return res;
    }

    std::optional<TConfig::THttpUnistat> TConfig::ParseHttpUnistat(const NJson::TConfig& config, const TString& jsonPoint) {
        if (!config.Contains(jsonPoint + "/unistat")) {
            return {};
        }

        THttpUnistat res;

        res.ListenAddress = config.As<TString>(jsonPoint + "/unistat/listen_address", "localhost");
        res.Port = config.As<ui32>(jsonPoint + "/unistat/port");
        res.Threads = config.As<size_t>(jsonPoint + "/unistat/threads", 2);
        res.MaxConnections = config.As<size_t>(jsonPoint + "/unistat/max_connections", 10);
        res.MaxQueueSize = config.As<size_t>(jsonPoint + "/unistat/max_queue_size", 10);
        res.MaxDelay = TDuration::MilliSeconds(config.As<size_t>(jsonPoint + "/unistat/max_delay", 1000));
        res.KeepAlive = config.As<bool>(jsonPoint + "/unistat/keep_alive", false);
        res.Path = config.As<TString>(jsonPoint + "/unistat/path");

        return res;
    }

    TConfig::TResponseTimeHeaders TConfig::ParseResponseTimeHeaders(const NJson::TConfig& config, const TString& jsonPoint) {
        TResponseTimeHeaders res;

        res.InQueue = config.As<TString>(jsonPoint + "/in_queue", "");
        res.InHandler = config.As<TString>(jsonPoint + "/in_handler", "");

        return res;
    }
}
