#include "daemon.h"

#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/log/registry.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <library/cpp/malloc/api/malloc.h>

#include <util/stream/file.h>
#include <util/system/fs.h>

namespace NPassport::NDaemon {
    // NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
    static THttpDaemon* GLOBAL_DAEMON = nullptr;
    // NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
    THttpDaemon::TSighupHandler SIGHUP_HANDLER;

    THttpDaemon::THttpDaemon(TConfig&& config)
        : Config_(std::move(config))
    {
        GLOBAL_DAEMON = this;

        InitMonitor();
        InitAllocator();
    }

    THttpDaemon::~THttpDaemon() {
        GLOBAL_DAEMON = nullptr;
    }

    void THttpDaemon::Start() {
        for (THttpInterface& i : HttpInterfaces_) {
            i.Start();
        }
        if (UnistatInterface_) {
            UnistatInterface_->Start();
        }

        for (THttpInterface& i : HttpInterfaces_) {
            i.Wait();
        }
        if (UnistatInterface_) {
            UnistatInterface_->Wait();
        }
    }

    void THttpDaemon::Stop() {
        for (THttpInterface& i : HttpInterfaces_) {
            i.Stop();
        }
        if (UnistatInterface_) {
            UnistatInterface_->Stop();
        }
    }

    static const TString DUMMY_MONITORING_RESPONSE =
        R"(<pool name="work_pool" threads="0" busy="0" queue="0" current_queue="0" all_tasks="0" clients="0" fail_queue="0"/>
)";
    TString THttpDaemon::Stats() const {
        TString res;
        res.reserve(DUMMY_MONITORING_RESPONSE.size() * HttpInterfaces_.size());
        for (const THttpInterface& i : HttpInterfaces_) {
            res.append(i.Stats());
        }
        return res;
    }

    void THttpDaemon::AddUnistat(NUnistat::TBuilder& builder) const {
        for (const THttpInterface& i : HttpInterfaces_) {
            i.AddUnistat(builder);
        }
    }

    void SignalsHandlers(int signo) {
        if (SIGTERM == signo || SIGINT == signo) {
            TLog::LogGotSignal((SIGTERM == signo ? TLog::ESignal::Sigterm : TLog::ESignal::Sigint));
            if (GLOBAL_DAEMON) {
                GLOBAL_DAEMON->Stop();
                GLOBAL_DAEMON = nullptr;
            }
        } else if (SIGHUP == signo) {
            TLog::LogGotSignal(TLog::ESignal::Sighup);
            if (GLOBAL_DAEMON) {
                NUtils::TLogRegistry::GetInstance().RotateLogs();
                SIGHUP_HANDLER();
            }
        }
    }

    void THttpDaemon::SetUpSignalsHandlers(TSighupHandler func) {
        SIGHUP_HANDLER = func;
        Y_ENSURE(signal(SIGTERM, SignalsHandlers) != SIG_ERR, "Cannot set up SIGTERM handler");
        Y_ENSURE(signal(SIGINT, SignalsHandlers) != SIG_ERR, "Cannot set up SIGINT handler");
        Y_ENSURE(signal(SIGHUP, SignalsHandlers) != SIG_ERR, "Cannot set up SIGHUP handler");
    }

    THttpServerOptions THttpDaemon::MakeHttpOptions(const TConfig::THttpBase& base,
                                                    ui16 port,
                                                    bool reusePort,
                                                    const TString& threadName) {
        THttpServerOptions o;

        o.AddBindAddress(base.ListenAddress, port);
        o.EnableReusePort(reusePort);
        o.SetThreads(base.Threads);
        o.SetMaxConnections(base.MaxConnections);
        o.SetMaxQueueSize(base.MaxQueueSize);
        o.EnableKeepAlive(base.KeepAlive);
        o.RequestsThreadName = threadName;

        return o;
    }

    void THttpDaemon::InitMonitor() {
        if (Config_.GetDaemon().MonitorPort) {
            Monitor_ = std::make_unique<TMonitor>(*this, *Config_.GetDaemon().MonitorPort);
        }
    }

    void THttpDaemon::InitAllocator() {
        if (!Config_.GetDaemon().EnableAllocatorDefrag) {
            bool result = NMalloc::MallocInfo().SetParam("EnableDefrag", "false");
            TLog::Debug() << "Trying to disable allocator defragmentation: "
                          << (result ? "SUCCESS" : "FAIL");
        }
    }
}
