#pragma once

#include "callback.h"
#include "config.h"
#include "http_interface.h"
#include "monitor.h"
#include "unistat.h"

#include <passport/infra/libs/cpp/xml/config.h>

#include <library/cpp/http/server/http.h>
#include <library/cpp/http/server/options.h>

#include <util/generic/string.h>

#include <atomic>
#include <memory>
#include <optional>

namespace NPassport::NDaemon::NPrivate {
    template <class T>
    class THasRotateLogs {
        using TYesType = char[1];
        using TNoType = char[2];

        template <typename C>
        static TYesType& Test(decltype(&C::RotateLogs));
        template <typename C>
        static TNoType& Test(...);

    public:
        static constexpr bool VALUE = sizeof(Test<T>(0)) == sizeof(TYesType);
    };
}

namespace NPassport::NDaemon {
    void SignalsHandlers(int);

    class THttpDaemon {
    public:
        using TSighupHandler = std::function<void()>;

        THttpDaemon(TConfig&& config);
        ~THttpDaemon();

        template <class T>
        void Init(T& s, const TString& name) {
            InitHttpCommon(s, name);
            InitHttpUnistat(s, name);

            SetUpSignalsHandlers([&s]() {
                if constexpr (NPrivate::THasRotateLogs<T>::VALUE) {
                    s.RotateLogs();
                } else {
                    Y_UNUSED(s);
                }
            });
        }

        void Start();
        void Stop();

        TString Stats() const;
        void AddUnistat(NUnistat::TBuilder& builder) const;

    private:
        static void SetUpSignalsHandlers(TSighupHandler func);
        static THttpServerOptions MakeHttpOptions(const TConfig::THttpBase& base, ui16 port, bool reusePort, const TString& threadName);

        template <class T>
        void InitHttpCommon(T& s, const TString& name) {
            for (const TConfig::THttpCommon::TInterface& p : Config_.GetHttpCommon().Interfaces) {
                for (size_t idx = 0; idx < p.ReuseCount; ++idx) {
                    HttpInterfaces_.push_back(THttpInterface::Create<T>(
                        s,
                        MakeHttpOptions(Config_.GetHttpCommon(), p.Port, p.ReuseCount > 1, name + "_worker"),
                        Config_.GetHttpCommon().MaxDelay,
                        Config_.GetHttpCommon().ResponseTimeHeaders,
                        idx + 1));
                }
            }
        }

        template <class T>
        void InitHttpUnistat(T& s, const TString& name) {
            if (!Config_.GetHttpUnistat()) {
                TLog::Info() << "Unistat interface disabled";
                return;
            }

            const TConfig::THttpUnistat& unistat = *Config_.GetHttpUnistat();

            UnistatHandler_ = std::make_unique<TUnistatHandler<T>>(
                s,
                [this](NUnistat::TBuilder& builder) { AddUnistat(builder); },
                unistat.Path);

            const THttpServerOptions opts = MakeHttpOptions(unistat, unistat.Port, false, name + "_unistat");
            UnistatInterface_ = std::make_unique<THttpInterface>(
                THttpInterface::Create<TBaseHandler>(
                    *UnistatHandler_,
                    opts,
                    unistat.MaxDelay));

            TLog::Info() << "Unistat interface started. port=" << opts.BindSockaddr.at(0).Port
                         << ". path=" << unistat.Path;
        }

        void InitMonitor();
        void InitAllocator();

    private:
        const TConfig Config_;

        // Should be last members
        std::vector<THttpInterface> HttpInterfaces_;
        std::unique_ptr<TBaseHandler> UnistatHandler_;
        std::unique_ptr<THttpInterface> UnistatInterface_;
        std::unique_ptr<TMonitor> Monitor_;
    };
}
