#include "monitor.h"

#include "daemon.h"

#include <util/generic/string.h>

namespace NPassport::NDaemon {
    TMonitor::TMonitor(const THttpDaemon& parent, int portNum)
        : Parent_(parent)
        , Stopping_(false)
    {
        Socket_ = socket(AF_INET, SOCK_STREAM, 0);
        if (-1 == Socket_) {
            throw yexception() << "Cannot create monitor socket";
        }

        int one = 1;
        if (-1 == setsockopt(Socket_, SOL_SOCKET, SO_REUSEADDR, &one, sizeof(one))) {
            throw yexception() << "Cannot reuse monitor port: " << errno;
        }

        sockaddr_in addr{};
        addr.sin_family = AF_INET;
        // NOLINTNEXTLINE(readability-isolate-declaration)
        addr.sin_port = htons(portNum);
        addr.sin_addr.s_addr = INADDR_ANY;
        memset(addr.sin_zero, '\0', sizeof(addr.sin_zero));

        if (-1 == bind(Socket_, (const sockaddr*)&addr, sizeof(sockaddr_in))) {
            throw yexception() << "Cannot bind monitor port address";
        }

        if (1 == listen(Socket_, SOMAXCONN)) {
            throw yexception() << "Cannot listen monitor port";
        }

        Thread_ = std::thread([this]() { Worker(); });
    }

    TMonitor::~TMonitor() {
        Stopping_.store(true);

        if (Thread_.joinable()) {
            shutdown(Socket_, SHUT_RD);
            Thread_.join();
        }

        if (-1 != Socket_) {
            close(Socket_);
        }
    }

    void TMonitor::Worker() {
        while (!IsStopping()) {
            int s = -1;
            try {
                s = accept(Socket_, nullptr, nullptr);
                if (IsStopping()) {
                    return;
                }
                if (-1 == s) {
                    throw yexception() << "Cannot accept connection on monitor port";
                }

                char buf[8];
                int rlen = read(s, buf, sizeof(buf));
                if (rlen <= 0) {
                    close(s);
                    continue;
                }

                const char c = buf[0];
                if ('i' == c || 'I' == c) {
                    TString info = Parent_.Stats();
                    write(s, info.c_str(), info.size());
                }

                close(s);
            } catch (const std::exception& e) {
                if (-1 != s) {
                    close(s);
                }
                Cerr << "Failed to process monitoring request: " << e.what()
                     << ", errno = " << errno
                     << Endl;
            }
        }
    }

    bool TMonitor::IsStopping() const {
        return Stopping_.load(std::memory_order_relaxed);
    }

}
